// JavaScript
// シーンのセットアップ
const scene = new THREE.Scene();
const camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
renderer.setClearColor(0x000000);
document.body.appendChild(renderer.domElement);

// カメラの初期位置
camera.position.set(0, 5, 0);

// ライト
const ambientLight = new THREE.AmbientLight(0xffffff, 0.5);
scene.add(ambientLight);
const directionalLight = new THREE.DirectionalLight(0xffffff, 0.5);
directionalLight.position.set(5, 10, 7.5);
scene.add(directionalLight);

// 立方体の生成
const cubeGeometry = new THREE.BoxGeometry(1, 1, 1);
const cubeMaterial = new THREE.MeshPhongMaterial({ 
    color: 0xffffff,
    specular: 0x555555,
    shininess: 10
});

const cubes = [];
for (let i = 0; i < 520; i++) {
    const cube = new THREE.Mesh(cubeGeometry, cubeMaterial);
    cube.position.set(
        Math.random() * 40 - 20,
        Math.random() * 20,
        Math.random() * 40 - 12
    );
    scene.add(cube);
    cubes.push(cube);
}

// 床
const floorGeometry = new THREE.PlaneGeometry(150, 150);
const floorMaterial = new THREE.MeshPhongMaterial({ 
    color: 0x111111,
    specular: 0x333333
});
const floor = new THREE.Mesh(floorGeometry, floorMaterial);
floor.rotation.x = -Math.PI / 2;
floor.position.y = -0.5;
scene.add(floor);

// 移動制御
let moveForward = false;
let moveBackward = false;
let moveLeft = false;
let moveRight = false;
let canJump = false;
let velocity = new THREE.Vector3();
const direction = new THREE.Vector3();

const speed = 0.01;
const jumpSpeed = 0.3;
const gravity = 0.01;

/* オイラー角　（視点制御） */
camera.rotation.order = 'YXZ'; /* デフォルトのXYZから変更 */
document.addEventListener('mousemove', (event) => {
    if (document.pointerLockElement === document.body) {
        const sensitivity = 0.003;
        camera.rotation.y -= event.movementX * sensitivity;
        camera.rotation.x -= event.movementY * sensitivity;
        camera.rotation.x = Math.max(-Math.PI / 2, Math.min(Math.PI / 2, camera.rotation.x));
    }
});

/* キー入力 */
document.addEventListener('keydown', (event) => {
    switch (event.code) {
        case 'KeyW': moveForward = true; break;
        case 'KeyS': moveBackward = true; break;
        case 'KeyA': moveRight = true; break;
        case 'KeyD': moveLeft = true; break;
        case 'Space': if (canJump) { velocity.y = jumpSpeed; canJump = false; } break;
    }
});

document.addEventListener('keyup', (event) => {
    switch (event.code) {
        case 'KeyW': moveForward = false; break;
        case 'KeyS': moveBackward = false; break;
        case 'KeyA': moveRight = false; break;
        case 'KeyD': moveLeft = false; break;
    }
});

// ウィンドウリサイズ対応
window.addEventListener('resize', () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
});

// オーディオのセットアップ
const audioListener = new THREE.AudioListener();
camera.add(audioListener);

// 足音用オーディオ
const footstepSound = new THREE.Audio(audioListener);
const audioLoader = new THREE.AudioLoader();
audioLoader.load('./audio/footstep.mp3', (buffer) => {
    footstepSound.setBuffer(buffer);
    footstepSound.setLoop(false);
    footstepSound.setVolume(2.5);
    console.log('Footstep sound loaded successfully');
}, undefined, (error) => {
    console.error('Error loading footstep sound:', error);
});

// BGM用オーディオ
const bgmSound = new THREE.Audio(audioListener);
let isBgmLoaded = false; // BGMの読み込み状態を管理

// audioLoader.load('./audio/Theme2.m4a', (buffer) => { /* 別のBGM */
audioLoader.load('./audio/bgm.mp3', (buffer) => {
    bgmSound.setBuffer(buffer);
    bgmSound.setLoop(true);
    bgmSound.setVolume(1.0);
    isBgmLoaded = true; // 読み込み完了フラグを立てる
    console.log('BGM loaded successfully');
    // クリック済みなら即再生
    if (audioListener.context.state === 'running' && !bgmSound.isPlaying) {
        bgmSound.play();
        console.log('BGM started after load');
    }
}, undefined, (error) => {
    console.error('Error loading BGM:', error);
});

// 足音再生の制御用変数
let lastFootstepTime = 0;
const footstepInterval = 0.35;

// 初回クリックでオーディオコンテキストを有効化
document.addEventListener('click', () => {
    document.body.requestPointerLock();
    if (audioListener.context.state !== 'running') {
        audioListener.context.resume().then(() => {
            console.log('Audio context resumed');
            if (isBgmLoaded && !bgmSound.isPlaying) {
                bgmSound.play();
                console.log('BGM started');
            }
        });
    } else if (isBgmLoaded && !bgmSound.isPlaying) {
        bgmSound.play();
        console.log('BGM started');
    }
});

const enemies = [];

// 敵を生成する関数
function spawnEnemy() {
    const position = new THREE.Vector3(
        Math.random() * 40 - 20, // X
        1,                      // Y（地面の高さに合わせる）
        Math.random() * 40 - 20  // Z
    );
    const enemy = new Enemy(scene, camera, position);
    enemies.push(enemy);
}
// スプラッシュエフェクト用のテクスチャを読み込む
const splashTextureLoader = new THREE.TextureLoader();
const splashTexture = splashTextureLoader.load('./img/splash-teki.png'); // 前回のスプラッシュ画像のパス

// スプラッシュエフェクトを表示する関数
function showSplashEffectOnEnemy(enemySprite) {
    const splashMaterial = new THREE.SpriteMaterial({ map: splashTexture, transparent: true });
    const splashSprite = new THREE.Sprite(splashMaterial);
    splashSprite.position.copy(enemySprite.position); // 敵の位置に配置
    splashSprite.position.y += 1; // 敵の中心より少し上に配置
    splashSprite.scale.set(2, 2, 1); // サイズ調整
    scene.add(splashSprite);

    // フェードアウトと削除
    let opacity = 1;
    const fadeOut = setInterval(() => {
        opacity -= 0.1;
        splashMaterial.opacity = opacity;
        if (opacity <= 0) {
            clearInterval(fadeOut);
            scene.remove(splashSprite);
            splashMaterial.dispose();
        }
    }, 30); // 0.3秒でフェードアウト（30ms間隔で10ステップ）
}
// 重低音用のオーディオ
const impactSound = new THREE.Audio(audioListener);
audioLoader.load('./audio/gun.mp3', (buffer) => {
    impactSound.setBuffer(buffer);
    impactSound.setLoop(false);
    impactSound.setVolume(1.2);
    console.log('Impact sound loaded successfully');
}, undefined, (error) => {
    console.error('Error loading impact sound:', error);
});
// 画面揺れ用の関数
function shakeScreen() {
    const shakeDuration = 150; // 揺れる時間（ミリ秒）
    const shakeIntensity = 0.5; // 揺れの強さ
    const originalPosition = camera.position.clone();

    let shakeTime = 0;
    const shakeInterval = setInterval(() => {
        shakeTime += 16; // 約60fpsで更新
        if (shakeTime >= shakeDuration) {
            clearInterval(shakeInterval);
            camera.position.copy(originalPosition); // 元の位置に戻す
            return;
        }

        // ランダムな揺れ
        const offsetX = (Math.random() - 0.5) * shakeIntensity;
        const offsetY = (Math.random() - 0.5) * shakeIntensity;
        camera.position.set(
            originalPosition.x + offsetX,
            originalPosition.y + offsetY,
            originalPosition.z
        );
    }, 16);
}

// 定期的に敵を生成（例: 5秒ごと）
setInterval(spawnEnemy, 3500);
const raycaster = new THREE.Raycaster();
const mouse = new THREE.Vector2();

document.addEventListener('mousedown', () => {
    if (document.pointerLockElement !== document.body) return;

    // 画面揺れと重低音を再生
    shakeScreen();
    if (impactSound.isPlaying) {
        impactSound.stop();
    }
    impactSound.play();

    // レイキャストで敵を検出
    mouse.x = 0;
    mouse.y = 0;
    raycaster.setFromCamera(mouse, camera);

    const enemySprites = enemies.map(enemy => enemy.getSprite());
    const intersects = raycaster.intersectObjects(enemySprites);

    if (intersects.length > 0) {
        const hitSprite = intersects[0].object;
        const hitEnemy = enemies.find(enemy => enemy.getSprite() === hitSprite);
        if (hitEnemy) {
            // スプラッシュエフェクトを表示
            showSplashEffectOnEnemy(hitSprite);

            const isDestroyed = hitEnemy.takeDamage();
            if (isDestroyed) {
                enemies.splice(enemies.indexOf(hitEnemy), 1);
            }
        }
    }
});


// HPと回復の設定
let hp = 100; // 初期HP
let isRecovering = false;
const recoveryRate = 2; // 回復速度
const recoveryInterval = 2000; // 回復間隔（ミリ秒）

const jumpHeight = 0.3; // ジャンプできる高さ
const damageThreshold = jumpHeight * 1.1; // 落下ダメージが発生する閾値

const hpBar = document.getElementById('hpBar');
const gameOverText = document.getElementById('gameOver');
let isDamageSoundLoaded = false;

audioLoader.load('./audio/Blow9.mp3', (buffer) => {
// audioLoader.load('./audio/footstep.mp3', (buffer) => {
    damageSound.setBuffer(buffer);
    damageSound.setLoop(false);
    damageSound.setVolume(2.0);
    isDamageSoundLoaded = true;
    alert('Test damage sound loaded successfully');
}, undefined, (error) => {
    console.error('Error loading test damage sound:', error);
});

// ダメージエフェクト用の要素を取得
const damageEffect = document.getElementById('damageEffect');
const damageSplash = document.getElementById('damageSplash');

// ダメージエフェクトを表示する関数
function showDamageEffect() {
    // エフェクトを表示
    damageEffect.style.display = 'block';
    damageSplash.style.opacity = '1'; // 初期透明度をリセット

    // フェードアウトアニメーションを適用
    damageSplash.classList.remove('fade-out'); // クラスをリセット
    void damageSplash.offsetWidth; // アニメーションをリスタートさせるトリック
    damageSplash.classList.add('fade-out');

    // アニメーション終了後に非表示にする
    setTimeout(() => {
        damageEffect.style.display = 'none';
    }, 300); // 0.3秒後に非表示
}

function applyFallDamage() {
    const fallDamage = Math.abs(velocity.y) * 5;
    hp -= fallDamage;
    hp = Math.max(0, hp);
    // console.log('Damage applied:', fallDamage, 'HP:', hp);

    if (isDamageSoundLoaded) {
        if (damageSound.isPlaying) {
            damageSound.stop();
        }
        damageSound.play();
    }
    
    // ダメージエフェクトを表示
    showDamageEffect();

    updateHpBar();

    if (hp <= 0) {
        gameOver();
    }
}

let enemiesDefeated = 0; // 倒した敵の数をカウント
// ゲームオーバー処理
function gameOver() {
    gameOverText.style.display = 'block';
    document.getElementById('enemiesDefeatedCount').textContent = enemiesDefeated; // カウントを表示
    cancelAnimationFrame(animationFrame);
}

function updateHpBar() {
    // console.log('Updating HP bar:', hp);
    hpBar.style.width = `${hp}%`;
}

// 回復機能を開始する関数
function startRecovery() {
    if (isRecovering) return;

    isRecovering = true;
    const recoveryIntervalId = setInterval(() => {
        if (hp === 0 || hp >= 100) {
            clearInterval(recoveryIntervalId);
            isRecovering = false;
        } else {
            hp = Math.min(100, hp + recoveryRate);
            updateHpBar();
        }
    }, recoveryInterval);
}

// アニメーションループの拡張
function animate() {
    animationFrame = requestAnimationFrame(animate);


    const delta = 1 / 60; // 仮のデルタ時間（実際は正確なデルタを計算可能）

    // 落下ダメージ判定
    if (velocity.y < -damageThreshold) {
        applyFallDamage();
    }

    // HP回復処理を開始
    startRecovery();

    // 移動計算
    velocity.x -= velocity.x * 0.1;
    velocity.z -= velocity.z * 0.1;
    velocity.y -= gravity;

    direction.z = Number(moveForward) - Number(moveBackward);
    direction.x = Number(moveRight) - Number(moveLeft);
    direction.normalize();

    if (moveForward || moveBackward) velocity.z -= direction.z * speed;
    if (moveLeft || moveRight) velocity.x -= direction.x * speed;

    camera.translateX(velocity.x);
    camera.translateZ(velocity.z);
    camera.position.y += velocity.y;

    // 地面との衝突判定
    if (camera.position.y < 1) {
        velocity.y = 0;
        camera.position.y = 1;
        canJump = true;
    }

    // 立方体との衝突判定
    let isOnSurface = camera.position.y <= 1;
    cubes.forEach(cube => {
        if (!cube.geometry.boundingBox) {
            cube.geometry.computeBoundingBox();
        }
        const cubeBounds = cube.geometry.boundingBox.clone();
        cubeBounds.applyMatrix4(cube.matrixWorld);
        const cameraPosition = new THREE.Vector3(camera.position.x, camera.position.y - 1.5, camera.position.z);

        if (cubeBounds.containsPoint(cameraPosition)) {
            velocity.y = 0;
            camera.position.y = cube.position.y + 1.5;
            canJump = true;
            isOnSurface = true;
        }
    });

    // 足音の再生
    const currentTime = performance.now() / 1000;
    const isMoving = moveForward || moveBackward || moveLeft || moveRight;
    if (isMoving && isOnSurface && (currentTime - lastFootstepTime > footstepInterval)) {
        if (footstepSound.isPlaying) {
            footstepSound.stop();
            // console.log('Footstep stopped and restarting');
        }
        footstepSound.play();
        // console.log('Playing footstep at time:', currentTime);
        lastFootstepTime = currentTime;
    }

    // 立方体の回転アニメーション
    cubes.forEach(cube => {
        cube.rotation.x += 0;
        cube.rotation.y += 0;
    });

    // 敵の更新
    enemies.forEach(enemy => enemy.update(delta));

    renderer.render(scene, camera);
}

animate();