// シーン、カメラ、レンダラーの設定
const scene = new THREE.Scene();
const camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
document.body.appendChild(renderer.domElement);

// プレイヤーの設定
let moveForward = false;
let moveBackward = false;
let moveLeft = false;
let moveRight = false;
const velocity = new THREE.Vector3();
const direction = new THREE.Vector3();
const playerSpeed = 0.1;

// 迷路の立方体配置
const cubeGeometry = new THREE.BoxGeometry(1, 1, 1);
const whiteMaterial = new THREE.MeshBasicMaterial({ color: 0xffffff });
const blackMaterial = new THREE.MeshBasicMaterial({ color: 0x000000 });
const cubes = [];

// 迷路生成（シンプルなグリッドパターン）
for (let x = -5; x <= 5; x += 2) {
    for (let z = -5; z <= 5; z += 2) {
        if (Math.random() > 0.3) { // ランダムに立方体を配置
            const cube = new THREE.Mesh(cubeGeometry, 
                Math.random() > 0.5 ? whiteMaterial : blackMaterial);
            cube.position.set(x, 0.5, z);
            scene.add(cube);
            cubes.push(cube);
        }
    }
}

// 床の作成
const floorGeometry = new THREE.PlaneGeometry(20, 20);
const floorMaterial = new THREE.MeshBasicMaterial({ color: 0x333333 });
const floor = new THREE.Mesh(floorGeometry, floorMaterial);
floor.rotation.x = -Math.PI / 2;
scene.add(floor);

// カメラの初期位置
camera.position.y = 1;
camera.position.z = 5;

// マウス制御
document.body.requestPointerLock = document.body.requestPointerLock || document.body.mozRequestPointerLock;
document.addEventListener('click', () => document.body.requestPointerLock());
        
document.addEventListener('mousemove', (event) => {
    const movementX = event.movementX || event.mozMovementX || 0;
    camera.rotation.y -= movementX * 0.002;
});

// キー入力
document.addEventListener('keydown', (event) => {
    switch (event.code) {
        case 'KeyW': moveForward = true; break;
        case 'KeyS': moveBackward = true; break;
        case 'KeyA': moveLeft = true; break;
        case 'KeyD': moveRight = true; break;
    }
});

document.addEventListener('keyup', (event) => {
    switch (event.code) {
        case 'KeyW': moveForward = false; break;
        case 'KeyS': moveBackward = false; break;
        case 'KeyA': moveLeft = false; break;
        case 'KeyD': moveRight = false; break;
    }
});

// 衝突検出用のボックス
const playerBox = new THREE.Box3();

function animate() {
    requestAnimationFrame(animate);

    // 移動処理
    velocity.x = 0;
    velocity.z = 0;
    direction.z = Number(moveForward) - Number(moveBackward);
    direction.x = Number(moveRight) - Number(moveLeft);
    irection.normalize();

    if (moveForward || moveBackward) velocity.z -= direction.z * playerSpeed;
    if (moveLeft || moveRight) velocity.x -= direction.x * playerSpeed;

    // 次の位置を計算
    const nextPosition = camera.position.clone();
    nextPosition.x += velocity.x;
    nextPosition.z += velocity.z;

    // 衝突検出
    playerBox.setFromCenterAndSize(
        nextPosition,
        new THREE.Vector3(0.5, 1, 0.5)
    );

    let collision = false;
    for (const cube of cubes) {
    const cubeBox = new THREE.Box3().setFromObject(cube);
        if (playerBox.intersectsBox(cubeBox)) {
            collision = true;
            break;
        }
    }

    // 衝突がない場合のみ移動
    if (!collision) {
        camera.position.x = nextPosition.x;
        camera.position.z = nextPosition.z;
    }

    renderer.render(scene, camera);
}

// ウィンドウリサイズ対応
window.addEventListener('resize', () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
});

animate();