// 敵クラスの定義
class Enemy {
    constructor(scene, player, position) {
        this.scene = scene;
        this.player = player; // プレイヤー（カメラ）を参照
        this.hp = 5; // 敵のHP（5発で倒れる）
        this.speed = 0.06; // 移動速度
        this.attackDamage = 2; // プレイヤーに与えるダメージ
        this.attackCooldown = 1; // 攻撃のクールダウン（秒）
        this.lastAttackTime = 0; // 最後に攻撃した時間

        // ジャンプ関連のプロパティ
        this.velocity = new THREE.Vector3(0, 0, 0); // 速度ベクトル
        this.canJump = true; // ジャンプ可能かどうか
        this.jumpSpeed = 0.25; // ジャンプの初速（プレイヤーと同じ）
        this.gravity = 0.01; // 重力（プレイヤーと同じ）

        // スプライトの設定
        const textureLoader = new THREE.TextureLoader();
        const spriteTexture = textureLoader.load('./img/アニメーション.png');
        this.spriteMaterial = new THREE.SpriteMaterial({ map: spriteTexture });
        this.sprite = new THREE.Sprite(this.spriteMaterial);
        this.sprite.position.copy(position);
        this.sprite.scale.set(2, 2, 1); // サイズ調整
        this.scene.add(this.sprite);

        // アニメーション設定
        this.frameCount = 6; // フレーム数（スプライトシートのフレーム数）
        this.currentFrame = 0;
        this.frameDuration = 0.07; // 各フレームの表示時間（秒）
        this.lastFrameUpdate = 0;

        // スプライトシートのフレームサイズ（UVマッピング）
        this.spriteMaterial.map.repeat.set(1 / this.frameCount, 1);
        this.spriteMaterial.map.offset.set(0, 0);

        // ランダムジャンプ用のタイマー
        this.lastRandomJump = 0;
        this.randomJumpInterval = Math.random() * 3 + 1; // 1〜4秒ごとにランダムジャンプ
    }

    jump() {
        if (this.canJump) {
            this.velocity.y = this.jumpSpeed;
            this.canJump = false;
        }
    }
    
    update(delta) {
        const currentTime = performance.now() / 1000;

        // プレイヤーに向かって移動
        const direction = this.player.position.clone().sub(this.sprite.position).normalize();
        this.sprite.position.x += direction.x * this.speed;
        this.sprite.position.z += direction.z * this.speed;

        // プレイヤーとの距離を計算
        const distanceToPlayer = this.sprite.position.distanceTo(this.player.position);

        // 攻撃（接触時）
        if (distanceToPlayer < 1.5 && currentTime - this.lastAttackTime > this.attackCooldown) {
            this.attack();
            this.lastAttackTime = currentTime;
        }

        // アニメーション更新
        if (currentTime - this.lastFrameUpdate > this.frameDuration) {
            this.currentFrame = (this.currentFrame + 1) % this.frameCount;
            this.spriteMaterial.map.offset.set(this.currentFrame / this.frameCount, 0);
            this.lastFrameUpdate = currentTime;
        }

        // プレイヤーの方を向く
        this.sprite.lookAt(this.player.position);

        // 重力とジャンプの処理
        this.velocity.y -= this.gravity;
        this.sprite.position.y += this.velocity.y;

        // 地面との衝突判定
        if (this.sprite.position.y < 1) {
            this.velocity.y = 0;
            this.sprite.position.y = 1;
            this.canJump = true;
        }

        // 立方体との衝突判定
        let isOnSurface = this.sprite.position.y <= 1;
        cubes.forEach(cube => {
            if (!cube.geometry.boundingBox) {
                cube.geometry.computeBoundingBox();
            }
            const cubeBounds = cube.geometry.boundingBox.clone();
            cubeBounds.applyMatrix4(cube.matrixWorld);
            const enemyPosition = new THREE.Vector3(this.sprite.position.x, this.sprite.position.y - 1, this.sprite.position.z);

            if (cubeBounds.containsPoint(enemyPosition)) {
                this.velocity.y = 0;
                this.sprite.position.y = cube.position.y + 1;
                this.canJump = true;
                isOnSurface = true;

                // 立方体の上にいる場合、ジャンプして乗り越える
                if (this.canJump && Math.random() < 0.05) { // 一定確率でジャンプ
                    this.jump();
                }
            }
        });

        // ランダムジャンプ
        if (currentTime - this.lastRandomJump > this.randomJumpInterval && this.canJump) {
            this.jump();
            this.lastRandomJump = currentTime;
            this.randomJumpInterval = Math.random() * 3 + 1; // 次のジャンプ間隔をランダムに設定
        }
    }

    attack() {
        hp -= this.attackDamage;
        hp = Math.max(0, hp);
        updateHpBar();

        // ダメージエフェクトを表示
        showDamageEffect();

        if (hp <= 0) {
            gameOver();
        }
    }

    takeDamage() {
        this.hp -= 1;
        if (this.hp <= 0) {
            this.scene.remove(this.sprite);
            enemiesDefeated++; // 敵を倒した数を増やす

            return true; // 敵が倒されたことを示す
        }
        return false;
    }

    getSprite() {
        return this.sprite;
    }
}