
// script.js
document.addEventListener('DOMContentLoaded', () => {
    // --- DOM Elements ---
    const eventList = document.getElementById('event-list');
    const addCommandBtn = document.getElementById('add-command-btn');
    const generateCodeBtn = document.getElementById('generate-code-btn');
    const copyCodeBtn = document.getElementById('copy-code-btn');
    const outputCode = document.getElementById('output-code');
    const modal = document.getElementById('command-modal');
    const modalTitle = document.getElementById('modal-title');
    const commandTypeSelect = document.getElementById('command-type');
    const modalOptions = document.getElementById('modal-options');
    const saveModalBtn = document.getElementById('save-modal-btn');
    const cancelModalBtn = document.getElementById('cancel-modal-btn');
    const copyFeedback = document.getElementById('copy-feedback');
    const importHtmlButton = document.getElementById('importHtmlButton');
    const importModal = document.getElementById('import-modal');
    const htmlText = document.getElementById('html-text');
    const htmlFile = document.getElementById('html-file');
    const importBtn = document.getElementById('import-btn');
    const cancelImportBtn = document.getElementById('cancel-import-btn');
    const contextMenu = document.getElementById('context-menu');
    const contextCopy = document.getElementById('context-copy');
    const contextCut = document.getElementById('context-cut');
    const contextPaste = document.getElementById('context-paste');

    // --- State ---
    let events = [];
    let editingPath = null; // null for new, array path for editing e.g., [0, 'then', 1]
    let importedIds = []; // インポートされたIDのリスト
    let selectedPaths = []; // 選択されたコマンドのパス
    let clipboard = []; // コピー/切り取りされたイベント
    let lastClickedPath = null;
    let isShiftPressed = false;

    // --- Keyboard Events ---
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Shift') isShiftPressed = true;
    });
    document.addEventListener('keyup', (e) => {
        if (e.key === 'Shift') isShiftPressed = false;
    });

    // --- Import Modal Logic ---
    const openImportModal = () => {
        importModal.classList.remove('hidden');
        htmlText.value = '';
        htmlFile.value = '';
    };

    const closeImportModal = () => {
        importModal.classList.add('hidden');
    };

    const processHtml = (htmlContent) => {
        const parser = new DOMParser();
        const doc = parser.parseFromString(htmlContent, 'text/html');
        const elementsWithId = doc.querySelectorAll('[id]');
        importedIds = Array.from(new Set(Array.from(elementsWithId).map(el => el.id))); // ユニークなIDリスト
        alert(`インポート完了: ${importedIds.length} 個のIDが見つかりました。`);
    };

    importHtmlButton.addEventListener('click', openImportModal);
    cancelImportBtn.addEventListener('click', closeImportModal);
    importBtn.addEventListener('click', () => {
        if (htmlFile.files.length > 0) {
            const file = htmlFile.files[0];
            const reader = new FileReader();
            reader.onload = (e) => {
                processHtml(e.target.result);
                closeImportModal();
            };
            reader.readAsText(file);
        } else if (htmlText.value.trim() !== '') {
            processHtml(htmlText.value);
            closeImportModal();
        } else {
            alert('HTMLコードを入力するか、ファイルを選択してください。');
        }
    });

    // --- Modal Logic ---
    const openModal = (path = null, defaultType = 'showMessage') => {
        editingPath = path;
        let command = {};
        if (path) {
            command = getEventByPath(path);
            modalTitle.textContent = 'コマンドの編集';
            commandTypeSelect.value = command.type;
        } else {
            modalTitle.textContent = 'コマンドの追加';
            commandTypeSelect.value = defaultType;
        }
        updateModalOptions(commandTypeSelect.value, command);
        modal.classList.remove('hidden');
    };
    const closeModal = () => {
        modal.classList.add('hidden');
        editingPath = null;
    };
    const updateModalOptions = (type, data = {}) => {
        modalOptions.innerHTML = '';
        switch (type) {
            case 'showMessage':
                modalOptions.innerHTML = `
                    <div>
                        <label for="msg-text" class="block text-sm font-medium text-gray-300 mb-1">メッセージ</label>
                        <textarea id="msg-text" rows="3" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white resize-none focus:ring-2 focus:ring-blue-500 focus:outline-none">${data.text || ''}</textarea>
                        <p class="text-xs text-gray-400 mt-1">変数を埋め込むには \`\${変数名}\` と入力します。(例: 所持金は\${gold}Gです)</p>
                    </div>
                `;
                break;
            case 'showAlert':
                modalOptions.innerHTML = `
                    <div>
                        <label for="alert-text" class="block text-sm font-medium text-gray-300 mb-1">アラートメッセージ</label>
                        <textarea id="alert-text" rows="3" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white resize-none focus:ring-2 focus:ring-blue-500 focus:outline-none">${data.text || ''}</textarea>
                        <p class="text-xs text-gray-400 mt-1">変数を埋め込むには \`\${変数名}\` と入力します。(例: HPが\${hp}になった)</p>
                    </div>
                `;
                break;
            case 'controlVariable':
                const isSwitch = data.value === 'true' || data.value === 'false';
                const isDom = data.valueType === 'dom';
                const initialValueType = data.valueType || (isSwitch ? 'switch' : (isDom ? 'dom' : 'value'));
                modalOptions.innerHTML = `
                    <div>
                        <label for="var-name" class="block text-sm font-medium text-gray-300 mb-1">変数名 (例: gold, hp, isDoorOpen)</label>
                        <input type="text" id="var-name" value="${data.variable || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="var-op" class="block text-sm font-medium text-gray-300 mb-1">操作</label>
                        <select id="var-op" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="=" ${data.operator === '=' ? 'selected' : ''}>= (代入)</option>
                            <option value="+=" ${data.operator === '+=' ? 'selected' : ''}>+= (加算)</option>
                            <option value="-=" ${data.operator === '-=' ? 'selected' : ''}>-= (減算)</option>
                            <option value="*=" ${data.operator === '*=' ? 'selected' : ''}>*= (乗算)</option>
                            <option value="/=" ${data.operator === '/=' ? 'selected' : ''}>/= (除算)</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">値の種類</label>
                        <div class="flex gap-x-6">
                            <label class="flex items-center cursor-pointer">
                                <input type="radio" name="value-type" value="value" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${initialValueType === 'value' ? 'checked' : ''}>
                                <span class="ml-2 text-gray-300">値</span>
                            </label>
                            <label class="flex items-center cursor-pointer">
                                <input type="radio" name="value-type" value="switch" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${initialValueType === 'switch' ? 'checked' : ''}>
                                <span class="ml-2 text-gray-300">スイッチ</span>
                            </label>
                            <label class="flex items-center cursor-pointer">
                                <input type="radio" name="value-type" value="dom" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${initialValueType === 'dom' ? 'checked' : ''}>
                                <span class="ml-2 text-gray-300">DOM操作</span>
                            </label>
                        </div>
                    </div>
                    <div id="var-value-container" class="mt-4"></div>
                `;
                const valueTypeRadios = modalOptions.querySelectorAll('input[name="value-type"]');
                const valueContainer = document.getElementById('var-value-container');
                
                const renderValueOptions = (type) => {
                    if (type === 'switch') {
                        const isOff = data.value === 'false';
                        valueContainer.innerHTML = `
                            <div class="flex gap-x-6">
                               <label class="flex items-center cursor-pointer">
                                   <input type="radio" name="switch-value" value="true" class="h-4 w-4 text-green-500 border-gray-500 focus:ring-green-500" ${!isOff ? 'checked' : ''}>
                                   <span class="ml-2 text-gray-300">ON</span>
                               </label>
                               <label class="flex items-center cursor-pointer">
                                   <input type="radio" name="switch-value" value="false" class="h-4 w-4 text-red-500 border-gray-500 focus:ring-red-500" ${isOff ? 'checked' : ''}>
                                   <span class="ml-2 text-gray-300">OFF</span>
                               </label>
                            </div>`;
                    } else if (type === 'dom') {
                        let optionsHtml = importedIds.map(id => `<option value="${id}" ${data.domId === id ? 'selected' : ''}>${id}</option>`).join('');
                        valueContainer.innerHTML = `
                            <label for="var-dom-id" class="block text-sm font-medium text-gray-300 mb-1">要素IDを選択</label>
                            <select id="var-dom-id" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                <option value="">-- 選択 --</option>
                                ${optionsHtml}
                            </select>
                            <p class="text-xs text-gray-400 mt-1">インポートされたIDから選択。手入力も可能。</p>
                            <input type="text" id="var-dom-id-custom" placeholder="カスタムID" value="${importedIds.includes(data.domId) ? '' : data.domId || ''}" class="mt-2 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                        `;
                    } else { // 'value'
                        valueContainer.innerHTML = `
                            <label for="var-val" class="block text-sm font-medium text-gray-300 mb-1">値 (例: 100, "こんにちは")</label>
                            <input type="text" id="var-val" value="${(initialValueType === 'value' && data.value) ? data.value : ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                        `;
                    }
                };
                valueTypeRadios.forEach(radio => {
                    radio.addEventListener('change', (e) => renderValueOptions(e.target.value));
                });
                renderValueOptions(initialValueType);
                break;
            case 'conditionalBranch':
                modalOptions.innerHTML = `
                    <div>
                        <label for="cond-type" class="block text-sm font-medium text-gray-300 mb-1">条件の種類</label>
                        <select id="cond-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="variable" ${data.conditionType !== 'switch' ? 'selected' : ''}>変数</option>
                            <option value="switch" ${data.conditionType === 'switch' ? 'selected' : ''}>スイッチ</option>
                        </select>
                    </div>
                    <div id="cond-options-container" class="mt-4"></div>
                `;
                const condTypeSelect = document.getElementById('cond-type');
                const renderCondOptions = (condType) => {
                    const container = document.getElementById('cond-options-container');
                    if (condType === 'switch') {
                        container.innerHTML = `
                            <label class="block text-sm font-medium text-gray-300 mb-1">スイッチの条件</label>
                            <div class="flex items-center gap-2">
                                <input type="text" id="cond-switch-var" placeholder="スイッチ変数名" value="${data.condition?.variable || ''}" class="flex-grow w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                <select id="cond-switch-val" class="bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="true" ${data.condition?.value !== 'false' ? 'selected' : ''}>がON</option>
                                    <option value="false" ${data.condition?.value === 'false' ? 'selected' : ''}>がOFF</option>
                                </select>
                            </div>`;
                    } else { // 'variable'
                        container.innerHTML = `
                            <label class="block text-sm font-medium text-gray-300 mb-1">変数の条件</label>
                            <div class="flex items-center gap-2">
                                <input type="text" id="cond-var" placeholder="変数名" value="${data.condition?.variable || ''}" class="flex-1 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                <select id="cond-op" class="bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="==" ${data.condition?.operator === '==' ? 'selected' : ''}>==</option>
                                    <option value="!=" ${data.condition?.operator === '!=' ? 'selected' : ''}>!=</option>
                                    <option value=">" ${data.condition?.operator === '>' ? 'selected' : ''}>&gt;</option>
                                    <option value=">=" ${data.condition?.operator === '>=' ? 'selected' : ''}>&gt;=</option>
                                    <option value="<" ${data.condition?.operator === '<' ? 'selected' : ''}>&lt;</option>
                                    <option value="<=" ${data.condition?.operator === '<=' ? 'selected' : ''}>&lt;=</option>
                                </select>
                                <input type="text" id="cond-val" placeholder="値" value="${data.condition?.value || ''}" class="flex-1 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            </div>`;
                    }
                };
                condTypeSelect.addEventListener('change', (e) => renderCondOptions(e.target.value));
                renderCondOptions(condTypeSelect.value);
                break;
            case 'loop':
                 modalOptions.innerHTML = `
                    <div>
                        <label for="loop-type" class="block text-sm font-medium text-gray-300 mb-1">ループの種類</label>
                        <select id="loop-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="count" ${data.loopType !== 'condition' ? 'selected' : ''}>回数</option>
                            <option value="condition" ${data.loopType === 'condition' ? 'selected' : ''}>条件</option>
                        </select>
                    </div>
                    <div id="loop-options-container" class="mt-4"></div>
                 `;
                 const loopTypeSelect = document.getElementById('loop-type');
                 const renderLoopOptions = (loopType) => {
                    const container = document.getElementById('loop-options-container');
                    if (loopType === 'condition') {
                        container.innerHTML = `
                            <div>
                                <label for="loop-cond-type" class="block text-sm font-medium text-gray-300 mb-1">条件の種類</label>
                                <select id="loop-cond-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="variable" ${data.conditionType !== 'switch' ? 'selected' : ''}>変数</option>
                                    <option value="switch" ${data.conditionType === 'switch' ? 'selected' : ''}>スイッチ</option>
                                </select>
                            </div>
                            <div id="loop-cond-options-container" class="mt-4"></div>
                        `;
                        const loopCondTypeSelect = document.getElementById('loop-cond-type');
                        const renderLoopCondOptions = (condType) => {
                            const condContainer = document.getElementById('loop-cond-options-container');
                            if (condType === 'switch') {
                                condContainer.innerHTML = `
                                    <label class="block text-sm font-medium text-gray-300 mb-1">スイッチの条件</label>
                                    <div class="flex items-center gap-2">
                                        <input type="text" id="loop-cond-switch-var" placeholder="スイッチ変数名" value="${data.condition?.variable || ''}" class="flex-grow w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                        <select id="loop-cond-switch-val" class="bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                            <option value="true" ${data.condition?.value !== 'false' ? 'selected' : ''}>がON</option>
                                            <option value="false" ${data.condition?.value === 'false' ? 'selected' : ''}>がOFF</option>
                                        </select>
                                    </div>`;
                            } else { // 'variable'
                                condContainer.innerHTML = `
                                    <label class="block text-sm font-medium text-gray-300 mb-1">変数の条件</label>
                                    <div class="flex items-center gap-2">
                                        <input type="text" id="loop-cond-var" placeholder="変数名" value="${data.condition?.variable || ''}" class="flex-1 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                        <select id="loop-cond-op" class="bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                            <option value="==" ${data.condition?.operator === '==' ? 'selected' : ''}>==</option>
                                            <option value="!=" ${data.condition?.operator === '!=' ? 'selected' : ''}>!=</option>
                                            <option value=">" ${data.condition?.operator === '>' ? 'selected' : ''}>&gt;</option>
                                            <option value=">=" ${data.condition?.operator === '>=' ? 'selected' : ''}>&gt;=</option>
                                            <option value="<" ${data.condition?.operator === '<' ? 'selected' : ''}>&lt;</option>
                                            <option value="<=" ${data.condition?.operator === '<=' ? 'selected' : ''}>&lt;=</option>
                                        </select>
                                        <input type="text" id="loop-cond-val" placeholder="値" value="${data.condition?.value || ''}" class="flex-1 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    </div>`;
                            }
                        };
                        loopCondTypeSelect.addEventListener('change', (e) => renderLoopCondOptions(e.target.value));
                        renderLoopCondOptions(loopCondTypeSelect.value);
                    } else { // 'count'
                        container.innerHTML = `
                            <label for="loop-count" class="block text-sm font-medium text-gray-300 mb-1">繰り返す回数</label>
                            <input type="number" id="loop-count" min="1" value="${data.count || 1}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                        `;
                    }
                };
                loopTypeSelect.addEventListener('change', (e) => renderLoopOptions(e.target.value));
                renderLoopOptions(loopTypeSelect.value);
                break;
            case 'loopBreak':
                // No options
                break;
            case 'defineFunction':
                modalOptions.innerHTML = `
                    <div>
                        <label for="func-name" class="block text-sm font-medium text-gray-300 mb-1">関数名</label>
                        <input type="text" id="func-name" value="${data.name || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="func-args" class="block text-sm font-medium text-gray-300 mb-1">引数 (カンマ区切り)</label>
                        <input type="text" id="func-args" value="${data.args ? data.args.join(', ') : ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                `;
                break;
            case 'callFunction':
                modalOptions.innerHTML = `
                    <div>
                        <label for="call-name" class="block text-sm font-medium text-gray-300 mb-1">関数名</label>
                        <input type="text" id="call-name" value="${data.name || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="call-args" class="block text-sm font-medium text-gray-300 mb-1">引数 (カンマ区切り)</label>
                        <input type="text" id="call-args" value="${data.args ? data.args.join(', ') : ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                `;
                break;
            case 'domOperation':
                modalOptions.innerHTML = `
                    <div>
                        <label for="dom-op-type" class="block text-sm font-medium text-gray-300 mb-1">操作の種類</label>
                        <select id="dom-op-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            <option value="eventListener" ${data.opType === 'eventListener' ? 'selected' : ''}>イベントリスナー</option>
                            <option value="manipulation" ${data.opType !== 'eventListener' ? 'selected' : ''}>要素操作</option>
                        </select>
                    </div>
                    <div id="dom-options-container" class="mt-4"></div>
                `;
                const domOpTypeSelect = document.getElementById('dom-op-type');
                const renderDomOptions = (opType) => {
                    const container = document.getElementById('dom-options-container');
                    if (opType === 'eventListener') {
                        container.innerHTML = `
                            <div>
                                <label for="dom-event-type" class="block text-sm font-medium text-gray-300 mb-1">イベントタイプ</label>
                                <select id="dom-event-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="click" ${data.eventType === 'click' ? 'selected' : ''}>click</option>
                                    <option value="mouseover" ${data.eventType === 'mouseover' ? 'selected' : ''}>mouseover</option>
                                    <option value="mouseout" ${data.eventType === 'mouseout' ? 'selected' : ''}>mouseout</option>
                                    <option value="mousemove" ${data.eventType === 'mousemove' ? 'selected' : ''}>mousemove</option>
                                    <option value="keydown" ${data.eventType === 'keydown' ? 'selected' : ''}>keydown</option>
                                    <option value="load" ${data.eventType === 'load' ? 'selected' : ''}>load</option>
                                    <option value="resize" ${data.eventType === 'resize' ? 'selected' : ''}>resize</option>
                                </select>
                            </div>
                            <div class="mt-4">
                                <label class="block text-sm font-medium text-gray-300 mb-2">ターゲット</label>
                                <div class="flex gap-x-6">
                                    <label class="flex items-center cursor-pointer">
                                        <input type="radio" name="dom-target-type" value="element" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${data.elementId ? 'checked' : ''}>
                                        <span class="ml-2 text-gray-300">要素ID</span>
                                    </label>
                                    <label class="flex items-center cursor-pointer">
                                        <input type="radio" name="dom-target-type" value="window" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${!data.elementId ? 'checked' : ''}>
                                        <span class="ml-2 text-gray-300">window</span>
                                    </label>
                                </div>
                            </div>
                            <div id="dom-element-container" class="mt-4" style="display: ${data.elementId ? 'block' : 'none'};">
                                <label for="dom-element-id" class="block text-sm font-medium text-gray-300 mb-1">要素IDを選択</label>
                                <select id="dom-element-id" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="">-- 選択 --</option>
                                    ${importedIds.map(id => `<option value="${id}" ${data.elementId === id ? 'selected' : ''}>${id}</option>`).join('')}
                                </select>
                                <p class="text-xs text-gray-400 mt-1">インポートされたIDから選択。手入力も可能。</p>
                                <input type="text" id="dom-element-id-custom" placeholder="カスタムID" value="${importedIds.includes(data.elementId) ? '' : data.elementId || ''}" class="mt-2 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                            </div>
                        `;
                        const targetRadios = container.querySelectorAll('input[name="dom-target-type"]');
                        const elementContainer = document.getElementById('dom-element-container');
                        targetRadios.forEach(radio => {
                            radio.addEventListener('change', (e) => {
                                elementContainer.style.display = e.target.value === 'element' ? 'block' : 'none';
                            });
                        });
                    } else { // manipulation
                        container.innerHTML = `
                            <div>
                                <label for="dom-manip-type" class="block text-sm font-medium text-gray-300 mb-1">操作タイプ</label>
                                <select id="dom-manip-type" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <option value="setText" ${data.manipType === 'setText' ? 'selected' : ''}>テキスト書き換え</option>
                                    <option value="addElement" ${data.manipType === 'addElement' ? 'selected' : ''}>要素追加</option>
                                    <option value="hide" ${data.manipType === 'hide' ? 'selected' : ''}>非表示</option>
                                    <option value="show" ${data.manipType === 'show' ? 'selected' : ''}>表示</option>
                                </select>
                            </div>
                            <div id="dom-manip-options" class="mt-4"></div>
                        `;
                        const manipTypeSelect = document.getElementById('dom-manip-type');
                        const renderManipOptions = (manipType) => {
                            const manipContainer = document.getElementById('dom-manip-options');
                            if (manipType === 'setText') {
                                let optionsHtml = importedIds.map(id => `<option value="${id}" ${data.elementId === id ? 'selected' : ''}>${id}</option>`).join('');
                                manipContainer.innerHTML = `
                                    <label for="dom-settext-id" class="block text-sm font-medium text-gray-300 mb-1">要素IDを選択</label>
                                    <select id="dom-settext-id" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                        <option value="">-- 選択 --</option>
                                        ${optionsHtml}
                                    </select>
                                    <input type="text" id="dom-settext-id-custom" placeholder="カスタムID" value="${importedIds.includes(data.elementId) ? '' : data.elementId || ''}" class="mt-2 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <label for="dom-settext-text" class="block text-sm font-medium text-gray-300 mt-4 mb-1">テキスト</label>
                                    <input type="text" id="dom-settext-text" value="${data.text || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                `;
                            } else if (manipType === 'addElement') {
                                let optionsHtml = importedIds.map(id => `<option value="${id}" ${data.parentId === id ? 'selected' : ''}>${id}</option>`).join('');
                                manipContainer.innerHTML = `
                                    <label for="dom-add-parent-id" class="block text-sm font-medium text-gray-300 mb-1">親要素IDを選択</label>
                                    <select id="dom-add-parent-id" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                        <option value="">-- 選択 --</option>
                                        ${optionsHtml}
                                    </select>
                                    <input type="text" id="dom-add-parent-id-custom" placeholder="カスタムID" value="${importedIds.includes(data.parentId) ? '' : data.parentId || ''}" class="mt-2 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                    <label for="dom-add-html" class="block text-sm font-medium text-gray-300 mt-4 mb-1">追加HTML</label>
                                    <textarea id="dom-add-html" rows="3" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white resize-none focus:ring-2 focus:ring-blue-500 focus:outline-none">${data.html || ''}</textarea>
                                `;
                            } else { // hide or show
                                let optionsHtml = importedIds.map(id => `<option value="${id}" ${data.elementId === id ? 'selected' : ''}>${id}</option>`).join('');
                                manipContainer.innerHTML = `
                                    <label for="dom-vis-id" class="block text-sm font-medium text-gray-300 mb-1">要素IDを選択</label>
                                    <select id="dom-vis-id" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                        <option value="">-- 選択 --</option>
                                        ${optionsHtml}
                                    </select>
                                    <input type="text" id="dom-vis-id-custom" placeholder="カスタムID" value="${importedIds.includes(data.elementId) ? '' : data.elementId || ''}" class="mt-2 w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                                `;
                            }
                        };
                        manipTypeSelect.addEventListener('change', (e) => renderManipOptions(e.target.value));
                        renderManipOptions(manipTypeSelect.value);
                    }
                };
                domOpTypeSelect.addEventListener('change', (e) => renderDomOptions(e.target.value));
                renderDomOptions(domOpTypeSelect.value);
                break;
            case 'defineConstant':
                modalOptions.innerHTML = `
                    <div>
                        <label for="const-name" class="block text-sm font-medium text-gray-300 mb-1">定数名</label>
                        <input type="text" id="const-name" value="${data.name || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="const-value" class="block text-sm font-medium text-gray-300 mb-1">値</label>
                        <input type="text" id="const-value" value="${data.value || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                `;
                break;
            case 'defineArray':
                modalOptions.innerHTML = `
                    <div>
                        <label for="array-name" class="block text-sm font-medium text-gray-300 mb-1">配列名</label>
                        <input type="text" id="array-name" value="${data.name || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="array-values" class="block text-sm font-medium text-gray-300 mb-1">値 (カンマ区切り)</label>
                        <input type="text" id="array-values" value="${data.values ? data.values.join(', ') : ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-300 mb-2">定義の種類</label>
                        <div class="flex gap-x-6">
                            <label class="flex items-center cursor-pointer">
                                <input type="radio" name="array-type" value="const" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${data.isConst !== false ? 'checked' : ''}>
                                <span class="ml-2 text-gray-300">const</span>
                            </label>
                            <label class="flex items-center cursor-pointer">
                                <input type="radio" name="array-type" value="let" class="h-4 w-4 text-blue-600 border-gray-500 focus:ring-blue-500" ${data.isConst === false ? 'checked' : ''}>
                                <span class="ml-2 text-gray-300">let</span>
                            </label>
                        </div>
                    </div>
                `;
                break;
            case 'returnValue':
                modalOptions.innerHTML = `
                    <div>
                        <label for="return-value" class="block text-sm font-medium text-gray-300 mb-1">戻り値</label>
                        <input type="text" id="return-value" value="${data.value || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                `;
                break;
            case 'comment':
                modalOptions.innerHTML = `
                    <div>
                        <label for="comment-text" class="block text-sm font-medium text-gray-300 mb-1">注釈テキスト</label>
                        <textarea id="comment-text" rows="3" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white resize-none focus:ring-2 focus:ring-blue-500 focus:outline-none">${data.text || ''}</textarea>
                    </div>
                `;
                break;
            case 'setInterval':
                modalOptions.innerHTML = `
                    <div>
                        <label for="interval-function" class="block text-sm font-medium text-gray-300 mb-1">呼び出す関数名</label>
                        <input type="text" id="interval-function" value="${data.functionName || ''}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                    <div>
                        <label for="interval-time" class="block text-sm font-medium text-gray-300 mb-1">実行間隔 (ミリ秒)</label>
                        <input type="number" id="interval-time" min="1" value="${data.interval || 1000}" class="w-full bg-gray-700 border border-gray-600 rounded-md p-2 text-white focus:ring-2 focus:ring-blue-500 focus:outline-none">
                    </div>
                `;
                break;
        }
    };
    commandTypeSelect.addEventListener('change', (e) => updateModalOptions(e.target.value));
    cancelModalBtn.addEventListener('click', closeModal);
    saveModalBtn.addEventListener('click', () => {
        const type = commandTypeSelect.value;
        const newEvent = { type };
        switch (type) {
            case 'showMessage':
                newEvent.text = document.getElementById('msg-text').value;
                break;
            case 'showAlert':
                newEvent.text = document.getElementById('alert-text').value;
                break;
            case 'controlVariable':
                newEvent.variable = document.getElementById('var-name').value;
                newEvent.operator = document.getElementById('var-op').value;
                const valueType = document.querySelector('input[name="value-type"]:checked')?.value;
                newEvent.valueType = valueType;
                if (valueType === 'switch') {
                    newEvent.value = document.querySelector('input[name="switch-value"]:checked')?.value;
                } else if (valueType === 'dom') {
                    const selectId = document.getElementById('var-dom-id').value;
                    const customId = document.getElementById('var-dom-id-custom').value.trim();
                    newEvent.domId = selectId || customId;
                } else { // value
                    newEvent.value = document.getElementById('var-val').value;
                }
                break;
            case 'conditionalBranch':
                const condType = document.getElementById('cond-type').value;
                newEvent.conditionType = condType;
                newEvent.then = editingPath ? getEventByPath(editingPath).then || [] : [];
                newEvent.else = editingPath ? getEventByPath(editingPath).else || [] : [];
                newEvent.condition = {};
                if (condType === 'switch') {
                    newEvent.condition.variable = document.getElementById('cond-switch-var').value;
                    newEvent.condition.value = document.getElementById('cond-switch-val').value;
                } else {
                    newEvent.condition.variable = document.getElementById('cond-var').value;
                    newEvent.condition.operator = document.getElementById('cond-op').value;
                    newEvent.condition.value = document.getElementById('cond-val').value;
                }
                break;
            case 'loop':
                const loopType = document.getElementById('loop-type').value;
                newEvent.loopType = loopType;
                newEvent.do = editingPath ? getEventByPath(editingPath).do || [] : [];
                if (loopType === 'condition') {
                    const condType = document.getElementById('loop-cond-type').value;
                    newEvent.conditionType = condType;
                    newEvent.condition = {};
                    if (condType === 'switch') {
                        newEvent.condition.variable = document.getElementById('loop-cond-switch-var').value;
                        newEvent.condition.value = document.getElementById('loop-cond-switch-val').value;
                    } else {
                        newEvent.condition.variable = document.getElementById('loop-cond-var').value;
                        newEvent.condition.operator = document.getElementById('loop-cond-op').value;
                        newEvent.condition.value = document.getElementById('loop-cond-val').value;
                    }
                } else {
                    newEvent.count = document.getElementById('loop-count').value;
                }
                break;
            case 'loopBreak':
                // No additional data
                break;
            case 'defineFunction':
                newEvent.name = document.getElementById('func-name').value;
                newEvent.args = document.getElementById('func-args').value.split(',').map(a => a.trim()).filter(a => a);
                newEvent.body = editingPath ? getEventByPath(editingPath).body || [] : [];
                break;
            case 'callFunction':
                newEvent.name = document.getElementById('call-name').value;
                newEvent.args = document.getElementById('call-args').value.split(',').map(a => a.trim()).filter(a => a);
                break;
            case 'domOperation':
                newEvent.opType = document.getElementById('dom-op-type').value;
                if (newEvent.opType === 'eventListener') {
                    newEvent.eventType = document.getElementById('dom-event-type').value;
                    const targetType = document.querySelector('input[name="dom-target-type"]:checked')?.value;
                    if (targetType === 'element') {
                        const selectId = document.getElementById('dom-element-id').value;
                        const customId = document.getElementById('dom-element-id-custom').value.trim();
                        newEvent.elementId = selectId || customId;
                    } else {
                        newEvent.elementId = null;
                    }
                    newEvent.body = editingPath ? getEventByPath(editingPath).body || [] : [];
                } else {
                    newEvent.manipType = document.getElementById('dom-manip-type').value;
                    if (newEvent.manipType === 'setText') {
                        const selectId = document.getElementById('dom-settext-id').value;
                        const customId = document.getElementById('dom-settext-id-custom').value.trim();
                        newEvent.elementId = selectId || customId;
                        newEvent.text = document.getElementById('dom-settext-text').value;
                    } else if (newEvent.manipType === 'addElement') {
                        const selectId = document.getElementById('dom-add-parent-id').value;
                        const customId = document.getElementById('dom-add-parent-id-custom').value.trim();
                        newEvent.parentId = selectId || customId;
                        newEvent.html = document.getElementById('dom-add-html').value;
                    } else { // 'hide' or 'show'
                        const selectId = document.getElementById('dom-vis-id').value;
                        const customId = document.getElementById('dom-vis-id-custom').value.trim();
                        newEvent.elementId = selectId || customId;
                    }
                }
                break;
            case 'defineConstant':
                newEvent.name = document.getElementById('const-name').value;
                newEvent.value = document.getElementById('const-value').value;
                break;
            case 'defineArray':
                newEvent.name = document.getElementById('array-name').value;
                newEvent.values = document.getElementById('array-values').value.split(',').map(v => v.trim()).filter(v => v);
                newEvent.isConst = document.querySelector('input[name="array-type"]:checked')?.value === 'const';
                break;
            case 'returnValue':
                newEvent.value = document.getElementById('return-value').value;
                break;
            case 'comment':
                newEvent.text = document.getElementById('comment-text').value;
                break;
            case 'setInterval':
                newEvent.functionName = document.getElementById('interval-function').value;
                newEvent.interval = document.getElementById('interval-time').value;
                break;
        }
        if (editingPath) {
            setEventByPath(editingPath, newEvent);
        } else {
            addEvent(newEvent, currentParentPath || null);
        }
        renderEvents();
        closeModal();
    });

    addCommandBtn.addEventListener('click', () => {
        currentParentPath = null;
        openModal();
    });

    let currentParentPath = null;

    // --- Path Utilities ---
    const getEventByPath = (path) => {
        let current = events;
        for (let p of path) {
            current = current[p];
        }
        return current;
    };

    const setEventByPath = (path, newEvent) => {
        let current = events;
        for (let i = 0; i < path.length - 1; i++) {
            current = current[path[i]];
        }
        current[path[path.length - 1]] = newEvent;
    };

    const deleteEventByPath = (path) => {
        let current = events;
        for (let i = 0; i < path.length - 1; i++) {
            current = current[path[i]];
        }
        current.splice(path[path.length - 1], 1);
        renderEvents();
    };

    const addEvent = (newEvent, parentPath = null) => {
        let targetArray;
        if (parentPath) {
            targetArray = getEventByPath(parentPath);
        } else {
            targetArray = events;
        }
        targetArray.push(newEvent);
        renderEvents();
    };

    const moveEvent = (sourcePath, targetPath, isNested = false) => {
        const event = getEventByPath(sourcePath);
        deleteEventByPath(sourcePath); // Remove from source
        let targetArray;
        if (isNested) {
            const targetEvent = getEventByPath(targetPath);
            if (targetEvent.type === 'conditionalBranch') {
                targetArray = targetEvent.then; // or else, but for simplicity, assuming 'then'
            } else if (targetEvent.type === 'loop') {
                targetArray = targetEvent.do;
            } else if (targetEvent.type === 'defineFunction' || targetEvent.type === 'domOperation' && targetEvent.opType === 'eventListener') {
                targetArray = targetEvent.body;
            }
        } else {
            targetArray = getEventByPath(targetPath.slice(0, -1)); // Parent array
            const index = targetPath[targetPath.length - 1] + 1; // Insert after
            targetArray.splice(index, 0, event);
            renderEvents();
            return;
        }
        targetArray.push(event);
        renderEvents();
    };

    const deepCopy = (obj) => {
        return JSON.parse(JSON.stringify(obj));
    };

    // --- Rendering ---
    const renderEvents = () => {
        eventList.innerHTML = renderEventList(events);
        setupDragAndDrop(); // Re-setup drag after render
    };

    const renderEventList = (eventArray, parentPath = []) => {
        if (!eventArray || eventArray.length === 0) return '';
        return eventArray.map((event, index) => {
            const path = [...parentPath, index];
            const pathStr = path.join(',');
            let content = '';
            let colorClass = '';
            let nestedContent = '';
            switch (event.type) {
            case 'showMessage':
                colorClass = 'color-message';
                content = `<strong>メッセージの表示:</strong> "${event.text}"`;
                break;
            case 'showAlert':
                colorClass = 'color-alert';
                content = `<strong>アラートの表示:</strong> "${event.text}"`;
                break;
            case 'controlVariable':
                colorClass = 'color-variable';
                const valueDisplay = event.valueType === 'dom' ? `document.getElementById('${event.domId}')` : event.value;
                content = `<strong>変数の操作:</strong> <span class="text-cyan-400 font-mono">${event.variable}</span> ${event.operator} <span class="text-purple-400 font-mono">${valueDisplay}</span>`;
                break;
            case 'conditionalBranch':
                colorClass = 'color-conditional';
                if (event.conditionType === 'switch') {
                    const status = event.condition.value === 'false' ? 'OFF' : 'ON';
                    content = `<strong>もし (スイッチ):</strong> <span class="text-cyan-400 font-mono">${event.condition.variable}</span> が <span class="text-purple-400 font-mono">${status}</span>`;
                } else {
                    content = `<strong>もし (変数):</strong> <span class="text-cyan-400 font-mono">${event.condition.variable}</span> ${event.condition.operator} <span class="text-purple-400 font-mono">${event.condition.value}</span>`;
                }
                nestedContent = `
                    <div class="event-command-nested mt-2 space-y-2">
                        ${renderEventList(event.then, [...path, 'then'])}
                        <button class="add-nested-btn bg-blue-800/50 hover:bg-blue-700/50 text-xs py-1 px-2 rounded" data-path="${[...path, 'then'].join(',')}">+ 追加</button>
                    </div>
                    <div class="event-command color-conditional p-2 mt-2">
                      <strong>そうでなければ:</strong>
                    </div>
                    <div class="event-command-nested mt-2 space-y-2">
                         ${renderEventList(event.else, [...path, 'else'])}
                         <button class="add-nested-btn bg-blue-800/50 hover:bg-blue-700/50 text-xs py-1 px-2 rounded" data-path="${[...path, 'else'].join(',')}">+ 追加</button>
                    </div>
                `;
                break;
            case 'loop':
                 colorClass = 'color-loop';
                if (event.loopType === 'condition') {
                   if (event.conditionType === 'switch') {
                       const status = event.condition.value === 'false' ? 'OFF' : 'ON';
                        content = `<strong>ループ (スイッチ):</strong> <span class="text-cyan-400 font-mono">${event.condition.variable}</span> が <span class="text-purple-400 font-mono">${status}</span> の間`;
                    } else {
                        content = `<strong>ループ (変数):</strong> <span class="text-cyan-400 font-mono">${event.condition.variable}</span> ${event.condition.operator} <span class="text-purple-400 font-mono">${event.condition.value}</span> の間`;
                    }
                 } else {
                   content = `<strong>ループ (回数):</strong> <span class="text-purple-400 font-mono">${event.count || 0}</span> 回繰り返す`;
                 }
                 nestedContent = `
                     <div class="event-command-nested mt-2 space-y-2">
                         ${renderEventList(event.do, [...path, 'do'])}
                         <button class="add-nested-btn bg-blue-800/50 hover:bg-blue-700/50 text-xs py-1 px-2 rounded" data-path="${[...path, 'do'].join(',')}">+ 追加</button>
                     </div>
                 `;
                break;
            case 'loopBreak':
                colorClass = 'color-break';
                content = `<strong>ループを中断</strong>`;
                break;
            case 'defineFunction':
                colorClass = 'color-function';
                content = `<strong>関数の定義:</strong> <span class="text-cyan-400 font-mono">${event.name}</span> (${event.args.join(', ')})`;
                nestedContent = `
                    <div class="event-command-nested mt-2 space-y-2">
                        ${renderEventList(event.body, [...path, 'body'])}
                        <button class="add-nested-btn bg-blue-800/50 hover:bg-blue-700/50 text-xs py-1 px-2 rounded" data-path="${[...path, 'body'].join(',')}">+ 追加</button>
                    </div>
                `;
                break;
            case 'callFunction':
                colorClass = 'color-call';
                content = `<strong>関数の呼び出し:</strong> <span class="text-cyan-400 font-mono">${event.name}</span> (${event.args.join(', ')})`;
                break;
            case 'domOperation':
                colorClass = 'color-dom';
                if (event.opType === 'eventListener') {
                    const target = event.elementId ? `ID: ${event.elementId}` : 'window';
                    content = `<strong>DOMイベント:</strong> ${event.eventType} on ${target}`;
                    nestedContent = `
                        <div class="event-command-nested mt-2 space-y-2">
                            ${renderEventList(event.body, [...path, 'body'])}
                            <button class="add-nested-btn bg-blue-800/50 hover:bg-blue-700/50 text-xs py-1 px-2 rounded" data-path="${[...path, 'body'].join(',')}">+ 追加</button>
                        </div>
                    `;
                } else {
                    let opDesc;
                    if (event.manipType === 'setText') opDesc = `テキスト書き換え ID: ${event.elementId} to "${event.text}"`;
                    else if (event.manipType === 'addElement') opDesc = `要素追加 to parent ${event.parentId}: ${event.html}`;
                    else if (event.manipType === 'hide') opDesc = `非表示 ID: ${event.elementId}`;
                    else opDesc = `表示 ID: ${event.elementId}`;
                    content = `<strong>DOM操作:</strong> ${opDesc}`;
                }
                break;
            case 'defineConstant':
                colorClass = 'color-constant';
                content = `<strong>定数の定義:</strong> <span class="text-cyan-400 font-mono">${event.name}</span> = <span class="text-purple-400 font-mono">${event.value}</span>`;
                break;
            case 'defineArray':
                colorClass = 'color-array';
                const decl = event.isConst ? 'const' : 'let';
                content = `<strong>配列の定義 (${decl}):</strong> <span class="text-cyan-400 font-mono">${event.name}</span> = [${event.values.join(', ')}]`;
                break;
            case 'returnValue':
                colorClass = 'color-return';
                content = `<strong>return:</strong> <span class="text-purple-400 font-mono">${event.value}</span>`;
                break;
            case 'comment':
                colorClass = 'color-comment';
                content = `<strong>注釈:</strong> "${event.text}"`;
                break;
            case 'setInterval':
                colorClass = 'color-interval';
                content = `<strong>インターバル:</strong> ${event.functionName} を ${event.interval}ms ごとに実行`;
                break;
        }
        const selectedClass = selectedPaths.some(p => JSON.stringify(p) === JSON.stringify(path)) ? 'selected' : '';
        return `
            <div class="event-command ${colorClass} p-3 rounded-lg flex justify-between items-center cursor-pointer ${selectedClass}" data-path="${pathStr}" draggable="true">
                <div class="flex-grow">${content}</div>
                <div class="flex-shrink-0 ml-4 space-x-2">
                    <button class="edit-btn text-blue-400 hover:text-blue-300" data-path="${pathStr}"><i class="fas fa-edit"></i></button>
                    <button class="delete-btn text-red-400 hover:text-red-300" data-path="${pathStr}"><i class="fas fa-trash"></i></button>
                </div>
            </div>
            ${nestedContent}
        `;
    }).join('');
    };

    // --- Selection Logic ---
    const selectCommand = (path, multi = false) => {
        if (!multi) {
            selectedPaths = [path];
        } else {
            const index = selectedPaths.findIndex(p => JSON.stringify(p) === JSON.stringify(path));
            if (index > -1) {
                selectedPaths.splice(index, 1);
            } else {
                selectedPaths.push(path);
            }
        }
        renderEvents();
    };

    // --- Context Menu Logic ---
    document.addEventListener('contextmenu', (e) => {
        e.preventDefault();
        const commandDiv = e.target.closest('.event-command');
        if (commandDiv) {
            const pathStr = commandDiv.dataset.path;
            const path = pathStr.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
            if (!selectedPaths.some(p => JSON.stringify(p) === JSON.stringify(path))) {
                selectCommand(path);
            }
            contextMenu.style.top = `${e.pageY}px`;
            contextMenu.style.left = `${e.pageX}px`;
            contextMenu.style.display = 'block';
        } else {
            contextMenu.style.display = 'none';
        }
    });

    document.addEventListener('click', () => {
        contextMenu.style.display = 'none';
    });

    contextCopy.addEventListener('click', () => {
        clipboard = selectedPaths.map(path => deepCopy(getEventByPath(path)));
        contextMenu.style.display = 'none';
    });

    contextCut.addEventListener('click', () => {
        clipboard = selectedPaths.map(path => deepCopy(getEventByPath(path)));
        selectedPaths.forEach(path => deleteEventByPath(path));
        selectedPaths = [];
        contextMenu.style.display = 'none';
        renderEvents();
    });

    contextPaste.addEventListener('click', () => {
        if (clipboard.length > 0 && selectedPaths.length > 0) {
            const targetPath = selectedPaths[0]; // Paste after the first selected
            const parentPath = targetPath.slice(0, -1);
            const index = targetPath[targetPath.length - 1] + 1;
            const parentArray = getEventByPath(parentPath);
            clipboard.forEach(clip => {
                parentArray.splice(index, 0, deepCopy(clip));
            });
            renderEvents();
        }
        contextMenu.style.display = 'none';
    });

    // --- Drag and Drop Setup ---
    const setupDragAndDrop = () => {
        const commands = document.querySelectorAll('.event-command');
        commands.forEach(cmd => {
            cmd.addEventListener('dragstart', (e) => {
                const pathStr = cmd.dataset.path;
                e.dataTransfer.setData('text/plain', pathStr);
            });

            cmd.addEventListener('dragover', (e) => {
                e.preventDefault();
            });

            cmd.addEventListener('drop', (e) => {
                e.preventDefault();
                const sourcePathStr = e.dataTransfer.getData('text/plain');
                const sourcePath = sourcePathStr.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
                const targetPathStr = cmd.dataset.path;
                const targetPath = targetPathStr.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
                const targetEvent = getEventByPath(targetPath);

                // Check if dropping on a nestable command
                if (['conditionalBranch', 'loop', 'defineFunction'].includes(targetEvent.type) || (targetEvent.type === 'domOperation' && targetEvent.opType === 'eventListener')) {
                    moveEvent(sourcePath, targetPath, true); // Move inside
                } else {
                    moveEvent(sourcePath, targetPath, false); // Move after
                }
            });
        });
    };

    // Event delegation for dynamically created buttons and selection
    eventList.addEventListener('click', (e) => {
        const commandDiv = e.target.closest('.event-command');
        if (commandDiv && !e.target.closest('button')) { // Click on command, not buttons
            const pathStr = commandDiv.dataset.path;
            const path = pathStr.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
            const multi = e.ctrlKey || e.metaKey || isShiftPressed;
            selectCommand(path, multi);
            if (isShiftPressed && lastClickedPath) {
                // Implement range selection if needed (advanced)
            }
            lastClickedPath = path;
        }

        const editBtn = e.target.closest('.edit-btn');
        if (editBtn) {
            const path = editBtn.dataset.path.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
            openModal(path);
        }
        const deleteBtn = e.target.closest('.delete-btn');
        if (deleteBtn) {
            if(confirm('このコマンドを削除しますか？')){
                const path = deleteBtn.dataset.path.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
                deleteEventByPath(path);
            }
        }
        
        const addNestedBtn = e.target.closest('.add-nested-btn');
        if(addNestedBtn) {
            currentParentPath = addNestedBtn.dataset.path.split(',').map(p => isNaN(parseInt(p)) ? p : parseInt(p));
            openModal();
        }
    });

    // --- Code Generation ---
    const generateCode = () => {
    let progress = document.getElementById("file");
    let currentValue = parseInt(progress.value);
    let intervalId;

    // 0からスタート
    if (currentValue === 100) {
        progress.value = 0;
        currentValue = 0;
    }

    // インターバルをクリアして新たに開始
    clearInterval(intervalId);

    // ランダムな増加量でvalueを更新
    intervalId = setInterval(() => {
        // ランダムな増加量（1～5の範囲など）
        let increment = Math.floor(Math.random() * 3.5) + 0.1;

        currentValue += increment;

        // valueが100を超えないように制限
        if (currentValue >= 100) {
            currentValue = 100;
            progress.value = currentValue;
            clearInterval(intervalId); // 100に達したらインターバルを停止
            // 次のsetIntervalでリセットされるので、ここでは何もしない
        } else {
            progress.value = currentValue;
        }
    }, 100); // 1秒ごとに更新（10秒で100になるように調整）
        const declaredVariables = new Set();
        const code = generateCodeRecursive(events, 0, declaredVariables);
        outputCode.textContent = code.trim();
        // const trimmedCode = code.trim();
        // const highlightedCode = colorizeCode(trimmedCode);
        // outputCode.innerHTML = highlightedCode;
    };

    const generateCodeRecursive = (eventArray, indentLevel, declaredVariables) => {
        const indent = '  '.repeat(indentLevel);
        if (!eventArray) return '';
        
        return eventArray.map(event => {
            let conditionString;
            switch (event.type) {
                case 'showMessage':
                    // Use template literals to allow variable embedding e.g. `Hello ${playerName}`
                    return `${indent}console.log(\`${event.text.replace(/`/g, '\\`')}\`);`;
                case 'showAlert':
                     // Use template literals to allow variable embedding
                    return `${indent}alert(\`${event.text.replace(/`/g, '\\`')}\`);`;
                case 'controlVariable':
                    let line = '';
                    if (!declaredVariables.has(event.variable)) {
                        line += `${indent}let ${event.variable};\n`;
                        declaredVariables.add(event.variable);
                    }
                    let valueStr = event.valueType === 'dom' ? `document.getElementById('${event.domId}')` : event.value;
                    line += `${indent}${event.variable} ${event.operator} ${valueStr};`;
                    return line;
                case 'conditionalBranch':
                    if (event.conditionType === 'switch') {
                        const varName = event.condition.variable;
                        conditionString = event.condition.value === 'false' ? `!${varName}` : varName;
                    } else {
                        conditionString = `${event.condition.variable} ${event.condition.operator} ${event.condition.value}`;
                    }
                    return `${indent}if (${conditionString}) {\n${generateCodeRecursive(event.then, indentLevel + 1, declaredVariables)}\n${indent}} else {\n${generateCodeRecursive(event.else, indentLevel + 1, declaredVariables)}\n${indent}}`;
                case 'loop':
                    if (event.loopType === 'condition') {
                        if (event.conditionType === 'switch') {
                            const varName = event.condition.variable;
                            conditionString = event.condition.value === 'false' ? `!${varName}` : varName;
                        } else {
                            conditionString = `${event.condition.variable} ${event.condition.operator} ${event.condition.value}`;
                        }
                        return `${indent}while (${conditionString}) {\n${generateCodeRecursive(event.do, indentLevel + 1, declaredVariables)}\n${indent}}`;
                    } else { // count
                        const i = `i_${indentLevel}`;
                        return `${indent}for (let ${i} = 0; ${i} < ${event.count}; ${i}++) {\n${generateCodeRecursive(event.do, indentLevel + 1, declaredVariables)}\n${indent}}`;
                    }
                case 'loopBreak':
                    return `${indent}break;`;
                case 'defineFunction':
                    const argsStr = event.args.join(', ');
                    return `${indent}const ${event.name} = (${argsStr}) => {\n${generateCodeRecursive(event.body, indentLevel + 1, declaredVariables)}\n${indent}}`;
                case 'callFunction':
                    const callArgsStr = event.args.join(', ');
                    return `${indent}${event.name}(${callArgsStr});`;
                case 'domOperation':
                    if (event.opType === 'eventListener') {
                        let target = event.elementId ? `document.getElementById('${event.elementId}')` : 'window';
                        if (event.eventType === 'mousemove') target = 'document'; // Adjust for mousemove
                        return `${indent}${target}.addEventListener('${event.eventType}', () => {\n${generateCodeRecursive(event.body, indentLevel + 1, declaredVariables)}\n${indent}});`;
                    } else { // manipulation
                        if (event.manipType === 'setText') {
                            return `${indent}document.getElementById('${event.elementId}').textContent = \`${event.text.replace(/`/g, '\\`')}\`;`;
                        } else if (event.manipType === 'addElement') {
                            return `${indent}document.getElementById('${event.parentId}').innerHTML += \`${event.html.replace(/`/g, '\\`')}\`;`;
                        } else if (event.manipType === 'hide') {
                            return `${indent}document.getElementById('${event.elementId}').style.display = 'none';`;
                        } else { // show
                            return `${indent}document.getElementById('${event.elementId}').style.display = 'block';`;
                        }
                    }
                case 'defineConstant':
                    return `${indent}const ${event.name} = ${event.value};`;
                case 'defineArray':
                    const decl = event.isConst ? 'const' : 'let';
                    return `${indent}${decl} ${event.name} = [${event.values.join(', ')}];`;
                case 'returnValue':
                    return `${indent}return ${event.value};`;
                case 'comment':
                    // code += `${indent}// ${event.text}\n`;
                    // break;
                    // return `${indent}// ${event.value};`;case 'comment':
                    // ユーザーが入力した注釈テキストが格納されているプロパティ（text）を参照するように修正
                    return `${indent}/* ${event.text || ''} */`;
                case 'setInterval':
                    return `${indent}setInterval(${event.functionName}, ${event.interval});`;
                default:
                    return `${indent}// Unknown command type: ${event.type}`;
            }
        }).join('\n');
    };

    generateCode();
    generateCodeBtn.addEventListener('click', generateCode);
    setInterval(generateCode, 10000);

    // --- Clipboard ---
    copyCodeBtn.addEventListener('click', () => {
        const textToCopy = outputCode.textContent;
        const textArea = document.createElement('textarea');
        textArea.value = textToCopy;
        document.body.appendChild(textArea);
        textArea.select();
        try {
            document.execCommand('copy');
            copyFeedback.textContent = 'クリップボードにコピーしました！';
            copyFeedback.style.backgroundColor = '#22c55e';
        } catch (err) {
            console.error('Copy failed', err);
            copyFeedback.textContent = 'コピーに失敗しました。';
            copyFeedback.style.backgroundColor = '#ef4444';
        }
        document.body.removeChild(textArea);
        copyFeedback.style.display = 'block';
        setTimeout(() => {
            copyFeedback.style.display = 'none';
        }, 2000);
    });
    // --- Initial Render ---
    renderEvents();
});