document.getElementById('start-button').addEventListener('click', startGame);
document.getElementById('continue-button').addEventListener('click', continueGame);
document.getElementById('save-button').addEventListener('click', saveGame);

let currentBGM = '';
let bgmPosition = 0;
let bgmElement = document.getElementById('bgm');

// 変数管理オブジェクト
let gameVariables = {
    scene1VisitedCount: 0,
};

// スイッチ管理オブジェクト
let switches = {
    'あ': false,
};

// 現在のシーンIDをグローバルに保持
let currentSceneId = 0;

// ゲーム開始時にセーブデータの有無を確認し、Continueボタンを有効化
window.onload = function() {
    const savedData = localStorage.getItem('gameSave');
    if (savedData) {
        document.getElementById('continue-button').disabled = false;
    }
};

function startGame() {
    document.getElementById('title-screen').style.display = 'none';
    document.getElementById('game-screen').style.display = 'block';
    showScene(0);
}

function continueGame() {
    const savedData = JSON.parse(localStorage.getItem('gameSave'));
    if (savedData) {
        document.getElementById('title-screen').style.display = 'none';
        document.getElementById('game-screen').style.display = 'block';
        gameVariables = savedData.gameVariables; // 変数の復元
        switches = savedData.switches; // スイッチの復元
        showScene(savedData.sceneId); // 保存されたシーンに移動
    }
}

function saveGame() {
    const saveData = {
        sceneId: currentSceneId, // グローバル変数からシーンIDを取得
        gameVariables: gameVariables,
        switches: switches
    };
    localStorage.setItem('gameSave', JSON.stringify(saveData));
    document.getElementById('continue-button').disabled = false;
    alert('ゲームが保存されました！');
}

const scenes = [
    {
        id: 0,
        background: './data/img/background', // 場面の背景画像
        defaultLeftCharacter: './data/img/character/', // 左側に表示する立ち絵
        defaultCenterCharacter: './data/img/character/', // 中央に表示する立ち絵
        defaultRightCharacter: './data/img/character/', // 右側に表示する立ち絵
        dialogue: 'これは最初のシーンです。', // 画面に表示するテキスト
        bgm: './data/bgm/', // BGM
        sfx: './data/se/', // 効果音
        choices: [
            { text: 'インデックス番号１', nextScene: 1 }, // ボタンを押した際、表示したいシーンのidとボタンに表示するテキスト
            { text: 'インデックス番号0', nextScene: 0 }
        ],
        alternateScenes: {
            'あ': {
                background: './data/img/background',
                defaultLeftCharacter: './data/img/character/', // プロパティ名を統一
                defaultCenterCharacter: './data/img/character/',
                defaultRightCharacter: './data/img/character/',
                dialogue: 'これはスイッチ「あ」がオンの場合のシーンです。',
                bgm: './data/bgm/',
                sfx: './data/se/',
                choices: [
                    { text: 'インデックス番号３', nextScene: 3 },
                    { text: 'インデックス番号０', nextScene: 0 }
                ]
            }
        }
    },
    {
        id: 1,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'インデックス番号１のシーンです。',
        bgm: './data/bgm/',
        sfx: './data/se/',
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号２', nextScene: 2 }
        ]
    },
    {
        id: 2,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'これはインデックス番号２のシーンです。',
        bgm: './data/bgm/',
        sfx: './data/se/',
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号０', nextScene: 0 }
        ]
    },
    {
        id: 3,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'ゲームオーバー',
        bgm: './data/bgm/',
        sfx: './data/se/',
        choices: [
            { text: 'インデックス番号０に戻る', nextScene: 0 },
        ]
    }
];

function showScene(index) {
    const scene = scenes.find(s => s.id === index);
    if (!scene) {
        console.error("Scene not found:", index);
        return;
    }

    // 現在のシーンIDを更新
    currentSceneId = scene.id;

    if (scene.id === 1) {
        gameVariables.scene1VisitedCount++;
    }

    if (gameVariables.scene1VisitedCount >= 5) {
        setSwitch('あ', true);
        gameVariables.scene1VisitedCount = 0;
    }

    if (scene.alternateScenes) {
        for (let key in scene.alternateScenes) {
            if (switches[key]) {
                const alternateScene = scene.alternateScenes[key];
                scene.background = alternateScene.background;
                scene.defaultLeftCharacter = alternateScene.defaultLeftCharacter;
                scene.defaultCenterCharacter = alternateScene.defaultCenterCharacter;
                scene.defaultRightCharacter = alternateScene.defaultRightCharacter;
                scene.dialogue = alternateScene.dialogue;
                scene.bgm = alternateScene.bgm;
                scene.sfx = alternateScene.sfx;
                scene.choices = alternateScene.choices;
                break;
            }
        }
    }

    document.getElementById('background').style.backgroundImage = `url(${scene.background})`;
    document.getElementById('left-character').style.backgroundImage = `url(${scene.defaultLeftCharacter})`;
    document.getElementById('center-character').style.backgroundImage = `url(${scene.defaultCenterCharacter})`;
    document.getElementById('right-character').style.backgroundImage = `url(${scene.defaultRightCharacter})`;
    changeBGM(scene.bgm);
    playSFX(scene.sfx);
    typeDialogue(scene.dialogue, () => {
        if (scene.choices.length > 0) {
            showChoices(scene.choices);
        }
    });
}

function setSwitch(switchName, state) {
    if (switches.hasOwnProperty(switchName)) {
        switches[switchName] = state;
    }
}

function changeBGM(bgm) {
    if (bgm === currentBGM) {
        bgmElement.currentTime = bgmPosition;
        bgmElement.play();
    } else {
        fadeOut(bgmElement, () => {
            bgmElement.src = bgm;
            bgmElement.currentTime = 0;
            bgmElement.play();
            fadeIn(bgmElement);
            currentBGM = bgm;
            bgmPosition = 0;
        });
    }
}

function fadeIn(audioElement) {
    let volume = 0;
    audioElement.volume = volume;
    const fadeInInterval = setInterval(() => {
        if (volume < 1) {
            volume += 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeInInterval);
        }
    }, 100);
}

function fadeOut(audioElement, callback) {
    let volume = audioElement.volume;
    const fadeOutInterval = setInterval(() => {
        if (volume > 0) {
            volume -= 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeOutInterval);
            callback();
        }
    }, 100);
}

function playSFX(sfx) {
    const sfxElement = new Audio(sfx);
    sfxElement.play();
}

bgmElement.addEventListener('timeupdate', () => {
    bgmPosition = bgmElement.currentTime;
});

function typeDialogue(text, callback) {
    const dialogueText = document.getElementById('dialogue-text');
    dialogueText.textContent = '';
    let i = 0;
    const typingSpeed = 50;
    const interval = setInterval(() => {
        if (i < text.length) {
            dialogueText.textContent += text.charAt(i);
            i++;
        } else {
            clearInterval(interval);
            setTimeout(callback, 500);
        }
    }, typingSpeed);
}

function showChoices(choices) {
    const choicesContainer = document.getElementById('choices');
    choicesContainer.style.display = 'block';
    choicesContainer.innerHTML = '';
    choices.forEach(choice => {
        const button = document.createElement('button');
        button.className = 'choice-button';
        button.textContent = choice.text;
        button.addEventListener('click', () => {
            choicesContainer.style.display = 'none';
            showScene(choice.nextScene);
        });
        choicesContainer.appendChild(button);
    });
}