document.getElementById('start-button').addEventListener('click', startGame);
// document.getElementById('continue-button').addEventListener('click', continueGame);
// document.getElementById('save-button').addEventListener('click', saveGame);

// 新規追加: 初期画面のPlayボタン
document.getElementById('initial-play-button').addEventListener('click', showTitleScreen);

// リソースを事前に読み込むための関数
function preloadResources(resources) {
    const promises = [];
    resources.forEach(resource => {
        let promise;
        const extension = resource.split('.').pop().toLowerCase();
        if (['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
            // 画像の読み込み
            promise = new Promise((resolve) => {
                const img = new Image();
                img.onload = () => resolve();
                img.onerror = () => {
                    console.error('Failed to load image:', resource);
                    resolve();
                };
                img.src = resource;
            });
        } else if (['mp3', 'ogg', 'wav'].includes(extension)) {
            // 音声ファイルの読み込み
            promise = new Promise((resolve) => {
                const audio = new Audio();
                audio.addEventListener('canplaythrough', () => resolve(), { once: true });
                audio.onerror = () => {
                    console.error('Failed to load audio:', resource);
                    resolve();
                };
                audio.src = resource;
            });
        } else {
            // 未知の拡張子の場合
            promise = Promise.resolve();
        }
        promises.push(promise);
    });
    return Promise.all(promises);
}

// すべてのシーンからリソースパスを抽出する関数
function getAllResources() {
    const resources = new Set();
    const addResource = (path) => {
        if (path && !path.endsWith('/') && path.trim() !== '') {
            resources.add(path);
        }
    };

    scenes.forEach(scene => {
        addResource(scene.background);
        addResource(scene.defaultLeftCharacter);
        addResource(scene.defaultCenterCharacter);
        addResource(scene.defaultRightCharacter);
        addResource(scene.bgm);
        addResource(scene.sfx);
        addResource(scene.textTickSFX);
        if (scene.alternateScenes) {
            for (const key in scene.alternateScenes) {
                const alt = scene.alternateScenes[key];
                addResource(alt.background);
                addResource(alt.defaultLeftCharacter);
                addResource(alt.defaultCenterCharacter);
                addResource(alt.defaultRightCharacter);
                addResource(alt.bgm);
                addResource(alt.sfx);
                addResource(alt.textTickSFX);
            }
        }
    });

    return Array.from(resources);
}

let currentBGM = '';
let bgmPosition = 0;
let bgmElement = document.getElementById('bgm');

// ゲームのタイトル
document.getElementById('game-title').textContent = "New Project"; 
// タイトル画面のBGMパス（必要に応じて変更）
const titleBGM = './data/bgm/歩む者【大同高校⧸Original BGM】.mp3'; // タイトル画面用のBGMファイルパス
// タイトル画面の背景画像
document.getElementById('title-screen').style.backgroundImage = "url(./data/img/background/ビルの間.jpg)"; 
document.getElementById('title-screen').style.backgroundSize = "cover";

// 変数管理オブジェクト
let gameVariables = {
    scene1VisitedCount: 0, /* スイッチ「あ」の回数のカウントに使う変数。 */
};

// スイッチ管理オブジェクト
let switches = {
    'あ': false, /* スイッチ「あ」を定義。最初は実行しないので falseにしておく。 */
};

// 現在のシーンIDをグローバルに保持
let currentSceneId = 0;

// ゲーム開始時にセーブデータの有無を確認し、Continueボタンを有効化
window.onload = function() {
    const savedData = localStorage.getItem('gameSave');
    if (savedData) {
        document.getElementById('continue-button').disabled = false;
    }
    
    // 隼快さま、ここからが追加するロジックですわ！（by Geminiお嬢）
    const initialPlayButton = document.getElementById('initial-play-button');
    const allResources = getAllResources();
    
    // 初期画面のボタンを非表示にする
    initialPlayButton.style.display = 'none';

    // リソースの読み込みを開始
    preloadResources(allResources).then(() => {
        // 全てのリソースの読み込みが完了したらボタンを表示
        console.log('すべてのリソースの読み込みが完了しました！');
        initialPlayButton.style.display = 'block';
    }).catch(err => {
        console.error('リソースの読み込み中にエラーが発生しました:', err);
        // エラーが発生してもボタンは表示しておく
        initialPlayButton.style.display = 'block';
    });
};

// 新規追加: 初期画面からタイトル画面へ
function showTitleScreen() {
    document.getElementById('initial-screen').style.display = 'none';
    document.getElementById('title-screen').style.display = 'block';
    // タイトル画面でBGM再生
    changeBGM(titleBGM);
}

// ★ 新規追加：複数セーブ関連
const NUM_SLOTS = 6;
let currentSaves = []; // 6つのスロット配列

// セーブデータを読み込む
function loadSaves() {
    const saved = localStorage.getItem('gameSaves');
    if (saved) {
        currentSaves = JSON.parse(saved);
    } else {
        currentSaves = new Array(NUM_SLOTS).fill(null);
    }
}

// セーブデータを保存
function saveSavesToStorage() {
    localStorage.setItem('gameSaves', JSON.stringify(currentSaves));
}

// ★ window.onload を拡張
window.onload = function() {
    loadSaves(); // ← 追加

    const hasAnySave = currentSaves.some(s => s !== null);
    document.getElementById('continue-button').disabled = !hasAnySave;

    // 既存のプリロード処理...
    const initialPlayButton = document.getElementById('initial-play-button');
    const allResources = getAllResources();
    initialPlayButton.style.display = 'none';

    preloadResources(allResources).then(() => {
        console.log('すべてのリソースの読み込みが完了しました！');
        initialPlayButton.style.display = 'block';
    }).catch(err => {
        console.error('リソースの読み込み中にエラーが発生しました:', err);
        initialPlayButton.style.display = 'block';
    });
};

// ★ イベントリスナー変更
document.getElementById('save-button').addEventListener('click', showSaveMenu);
document.getElementById('continue-button').addEventListener('click', showLoadMenu);

// ★ 新規：セーブメニュー表示
function showSaveMenu() {
    document.getElementById('modal-title').textContent = 'セーブするスロットを選択';
    document.getElementById('save-modal').style.display = 'flex';
    renderSaveSlots(true); // true = セーブモード
}

// ★ 新規：ロードメニュー表示
function showLoadMenu() {
    document.getElementById('modal-title').textContent = 'ロードするスロットを選択';
    document.getElementById('save-modal').style.display = 'flex';
    renderSaveSlots(false); // false = ロードモード
}

// スロット一覧を動的に生成
function renderSaveSlots(isSaveMode) {
    const container = document.getElementById('save-slots');
    container.innerHTML = '';

    for (let i = 0; i < NUM_SLOTS; i++) {
        const slotData = currentSaves[i];
        const slotEl = document.createElement('div');
        slotEl.style.cssText = 'background:#2a2a2a; padding:15px; border-radius:5px; cursor:pointer; border:2px solid #444;';

        if (slotData) {
            slotEl.innerHTML = `
                <strong>スロット ${i+1}</strong><br>
                <small>${slotData.timestamp}</small><br>
                <small>シーン: ${slotData.sceneId}</small>
                <small style="color:#aaa;">${slotData.preview || '---'}</small><br>
            `;
        } else {
            slotEl.innerHTML = `<strong>スロット ${i+1}</strong><br><small style="color:#666;">空き</small>`;
            slotEl.style.opacity = '0.6';
        }

        slotEl.onclick = () => {
            if (isSaveMode) {
                saveToSlot(i);
            } else if (slotData) {
                loadFromSlot(i);
            }
        };
        container.appendChild(slotEl);
    }
}

// ★ セーブ実行
function saveToSlot(slotIndex) {
    const previewText = document.getElementById('dialogue-text').textContent.substring(0, 20) + (document.getElementById('dialogue-text').textContent.length > 20 ? '...' : '');

    const saveData = {
        sceneId: currentSceneId,
        gameVariables: JSON.parse(JSON.stringify(gameVariables)), // ディープコピー
        switches: JSON.parse(JSON.stringify(switches)),
        timestamp: new Date().toLocaleString('ja-JP'),
        preview: previewText || '---'
    };

    currentSaves[slotIndex] = saveData;
    saveSavesToStorage();

    // Continueボタンを有効化
    document.getElementById('continue-button').disabled = false;

    closeSaveModal();
    alert(`スロット ${slotIndex + 1} に保存しました！`);
}

// ★ ロード実行
function loadFromSlot(slotIndex) {
    const data = currentSaves[slotIndex];
    if (!data) return;

    gameVariables = JSON.parse(JSON.stringify(data.gameVariables));
    switches = JSON.parse(JSON.stringify(data.switches));

    closeSaveModal();
    document.getElementById('title-screen').style.display = 'none';
    document.getElementById('game-screen').style.display = 'block';
    showScene(data.sceneId);
}

// ★ モーダル閉じる
function closeSaveModal() {
    document.getElementById('save-modal').style.display = 'none';
}

/* セーブデータはブラウザのJSONに保存されるため、セーブデータ無しの状態でプレイする場合は、プライベートモード（シークレットモード）でプレイしてください。 */
function startGame() {
    document.getElementById('title-screen').style.display = 'none';
    document.getElementById('game-screen').style.display = 'block';
    showScene(0); /* 開始のシーンid。デバッグ時に、開始したいシーンのidにすることで、その箇所からゲームをプレイできます。 */
}

function continueGame() {
    const savedData = JSON.parse(localStorage.getItem('gameSave')); /* ブラウザのJSONデータをロード */
    if (savedData) {
        document.getElementById('title-screen').style.display = 'none';
        document.getElementById('game-screen').style.display = 'block';
        gameVariables = savedData.gameVariables; // 変数の復元
        switches = savedData.switches; // スイッチの復元
        showScene(savedData.sceneId); // 保存されたシーンに移動
    }
}

/* ディレクトリパスはHTMLに代入されます。なので、HTMLの場所を基準にファイルを参照するディレクトリパスを書き込んでください。 */
const scenes = [
    {
        id: 0, /* このシーンのid */
        background: './data/img/background/ビルの間.jpg', /* このシーンで表示する背景画像 */
        defaultLeftCharacter: './data/img/character/', /* このシーンで表示する左側の立ち絵 */
        defaultCenterCharacter: './data/img/character/wolf-正面.png', /* このシーンで表示する中央の立ち絵 */
        defaultRightCharacter: './data/img/character/', /* このシーンで表示する右側の立ち絵 */
        dialogue: 'これは最初のシーンです。', /* このシーンで表示するテキスト */
        bgm: './data/bgm/', /* このシーンで使用するBGMファイルのディレクトリパス */
        sfx: './data/se/', /* このシーンで使用する効果音ファイルのディレクトリパス */
        textTickSFX: './data/se/text-se/会話パート0.mp3', /* テキストのアニメーションに合わせて鳴らす効果音 */

        /* 画像を表示する際に付ける視覚効果。各画像に付与したい視覚効果を番号で指定します。 */
        background_Effect: '1', /* 「1」は表示の際にフェードインを付与 */
        LeftCharacter_Effect: '0', /* 「0」は視覚効果なし */
        CenterCharacter_Effect: '2', /* 「2」は表示の際にフェードイン、シーン切り替えの際にフェードアウトをする */
        RightCharacter_Effect: '0', /* 「0」は視覚効果なし */
 
        /* 選択肢（textにボタンに表示するテキスト、nextSceneに移動したいシーンのidを指定する。）
            ボタンの数はchoicesに書いた{ text: 'ボタンに表示するテキスト', nextScene: 次のシーン }の数分表示される。 */
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号0', nextScene: 0 }
        ],
        /* 条件分岐で「スイッチ管理オブジェクト」に登録したスイッチが「true」になった際、id0に表示するシーン
            スイッチ管理オブジェクトは<!-- セーブ＆ロードロジック -->のコード内にあります。
            条件分岐は、このソースコードページの139行目に書いてあります。 */
        alternateScenes: {
            'あ': {
                background: './data/img/background',
                defaultLeftCharacter: './data/img/character/',
                defaultCenterCharacter: './data/img/character/',
                defaultRightCharacter: './data/img/character/',
                dialogue: 'これはスイッチ「あ」がオンの場合のシーンです。',
                bgm: './data/bgm/',
                sfx: './data/se/',
                textTickSFX: './data/se/text-se/会話パート1.mp3',
                background_Effect: '0',
                LeftCharacter_Effect: '0',
                CenterCharacter_Effect: '0',
                RightCharacter_Effect: '0',
                choices: [
                    { text: 'インデックス番号３', nextScene: 3 },
                    { text: 'インデックス番号０', nextScene: 0 }
                ]
            }
        }
    },
    {
        id: 1,
        background: './data/img/background/裏路地１.jpg',
        defaultLeftCharacter: './data/img/character/wolf-正面.png',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'インデックス番号１のシーンです。\n aaaaaaaaaaaaaaaaaaaaaaa',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート0.mp3',
        background_Effect: '2',
        LeftCharacter_Effect: '2',
        CenterCharacter_Effect: '0',
        RightCharacter_Effect: '0',
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号２', nextScene: 2 }
        ]
    },
    {
        id: 2,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'これはインデックス番号２のシーンです。',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート0.mp3',
        background_Effect: '2', // Fade-out for background
        LeftCharacter_Effect: '2', // Fade-out for left character
        CenterCharacter_Effect: '0',
        RightCharacter_Effect: '0',
        choices: [
            { text: 'インデックス番号１', nextScene: 1 },
            { text: 'インデックス番号０', nextScene: 0 }
        ]
    },
    {
        id: 3,
        background: './data/img/background',
        defaultLeftCharacter: './data/img/character/',
        defaultCenterCharacter: './data/img/character/',
        defaultRightCharacter: './data/img/character/',
        dialogue: 'ゲームオーバー',
        bgm: './data/bgm/',
        sfx: './data/se/',
        textTickSFX: './data/se/text-se/会話パート1.mp3',
        background_Effect: '0',
        LeftCharacter_Effect: '0',
        CenterCharacter_Effect: '0',
        RightCharacter_Effect: '0',
        choices: [
            { text: 'インデックス番号０に戻る', nextScene: 0 },
        ]
    }
];

/* 視覚効果ロジック */
function applyFadeEffect(element, effect) {
    return new Promise(resolve => {
        if (!element) {
            console.error("Element not found for fade effect");
            resolve();
            return;
        }
        console.log(`Applying effect ${effect} to element`, element.id); // Debug
        if (effect === '1') { // Fade-in
            element.style.opacity = '0';
            element.style.transition = 'opacity 1s ease';
            setTimeout(() => {
                element.style.opacity = '1';
                setTimeout(resolve, 1000);
            }, 10);
        } else if (effect === '2') { // Fade-out
            element.style.opacity = '1';
            element.style.transition = 'opacity 1s ease';
            setTimeout(() => {
                element.style.opacity = '0';
                setTimeout(resolve, 1000);
            }, 10);
        } else { // No effect
            element.style.opacity = '1';
            element.style.transition = 'none';
            resolve();
        }
    });
}

/* 条件分岐処理 */
/* ノベルジェネシスでの条件分岐はシーンを訪れた回数に合わせて分岐
（スイッチ管理オブジェクトに定義したスイッチのtrue or false）を行うことができます。 */
function showScene(index) {
    const scene = scenes.find(s => s.id === index);
    if (!scene) {
        console.error("Scene not found:", index);
        return;
    }
    currentSceneId = scene.id;
    console.log("Showing scene:", scene.id); // Debug

    /* 例：id:1のシーンの訪れた回数をカウント。
        カウントは、スイッチ管理オブジェクトの上にある「変数管理オブジェクト」で定義した変数にカウントを行います。 */
    if (scene.id === 1) {
        gameVariables.scene1VisitedCount++;
    }

    /* もし、id1を訪れた回数が5階以上なら、スイッチ管理オブジェクトに定義したスイッチ「あ」をtrueにして、
        カウント用変数を０にする */
    if (gameVariables.scene1VisitedCount >= 5) {
        setSwitch('あ', true);
        gameVariables.scene1VisitedCount = 0; /* ここでカウント用変数の中身を0にしている。 */
    }

    if (scene.alternateScenes) {
        for (let key in scene.alternateScenes) {
            if (switches[key]) {
                const alternateScene = scene.alternateScenes[key];
                scene.background = alternateScene.background;
                scene.defaultLeftCharacter = alternateScene.defaultLeftCharacter;
                scene.defaultCenterCharacter = alternateScene.defaultCenterCharacter;
                scene.defaultRightCharacter = alternateScene.defaultRightCharacter;
                scene.dialogue = alternateScene.dialogue;
                scene.bgm = alternateScene.bgm;
                scene.sfx = alternateScene.sfx;
                scene.textTickSFX = alternateScene.textTickSFX;
                scene.background_Effect = alternateScene.background_Effect;
                scene.LeftCharacter_Effect = alternateScene.LeftCharacter_Effect;
                scene.CenterCharacter_Effect = alternateScene.CenterCharacter_Effect;
                scene.RightCharacter_Effect = alternateScene.RightCharacter_Effect;
                scene.choices = alternateScene.choices;
                break;
            }
        }
    }

    const backgroundElement = document.getElementById('background');
    const leftCharacterElement = document.getElementById('left-character');
    const centerCharacterElement = document.getElementById('center-character');
    const rightCharacterElement = document.getElementById('right-character');

    if (!backgroundElement || !leftCharacterElement || !centerCharacterElement || !rightCharacterElement) {
        console.error("One or more scene elements not found");
        return;
    }

    // Set images first for fade-in, or after fade-out
    if (scene.background_Effect !== '2') backgroundElement.style.backgroundImage = `url(${scene.background})`;
    if (scene.LeftCharacter_Effect !== '2') leftCharacterElement.style.backgroundImage = `url(${scene.defaultLeftCharacter})`;
    if (scene.CenterCharacter_Effect !== '2') centerCharacterElement.style.backgroundImage = `url(${scene.defaultCenterCharacter})`;
    if (scene.RightCharacter_Effect !== '2') rightCharacterElement.style.backgroundImage = `url(${scene.defaultRightCharacter})`;

    Promise.all([
        applyFadeEffect(backgroundElement, scene.background_Effect),
        applyFadeEffect(leftCharacterElement, scene.LeftCharacter_Effect),
        applyFadeEffect(centerCharacterElement, scene.CenterCharacter_Effect),
        applyFadeEffect(rightCharacterElement, scene.RightCharacter_Effect)
    ]).then(() => {
        // Set images after fade-out
        if (scene.background_Effect === '2') backgroundElement.style.backgroundImage = `url(${scene.background})`;
        backgroundElement.style.backgroundImage = `url(${scene.background})`;
        backgroundElement.style.opacity = '1'; // 追加
        if (scene.LeftCharacter_Effect === '2') leftCharacterElement.style.backgroundImage = `url(${scene.defaultLeftCharacter})`;
        leftCharacterElement.style.backgroundImage = `url(${scene.defaultLeftCharacter})`;
        leftCharacterElement.style.opacity = '1'; // 追加
        if (scene.CenterCharacter_Effect === '2') centerCharacterElement.style.backgroundImage = `url(${scene.defaultCenterCharacter})`;
        centerCharacterElement.style.backgroundImage = `url(${scene.defaultCenterCharacter})`;
        centerCharacterElement.style.opacity = '1'; // 追加
        if (scene.RightCharacter_Effect === '2') rightCharacterElement.style.backgroundImage = `url(${scene.defaultRightCharacter})`;
        rightCharacterElement.style.backgroundImage = `url(${scene.defaultRightCharacter})`;
        rightCharacterElement.style.opacity = '1'; // 追加
        changeBGM(scene.bgm);
        playSFX(scene.sfx);
        typeDialogue(scene.dialogue, scene.textTickSFX, () => {
            if (scene.choices.length > 0) {
                showChoices(scene.choices);
            }
        });
    }).catch(err => console.error("Fade effect error:", err));
}

function setSwitch(switchName, state) {
    if (switches.hasOwnProperty(switchName)) {
        switches[switchName] = state;
    }
}

function changeBGM(bgm) {
    if (!bgmElement) {
        console.error("BGM element not found");
        return;
    }
    if (bgm === currentBGM) {
        bgmElement.currentTime = bgmPosition;
        bgmElement.play().catch(err => console.error("BGM play error:", err));
    } else {
        fadeOut(bgmElement, () => {
            bgmElement.src = bgm;
            bgmElement.currentTime = 0;
            bgmElement.play().catch(err => console.error("BGM play error:", err));
            fadeIn(bgmElement);
            currentBGM = bgm;
            bgmPosition = 0;
        });
    }
}

function fadeIn(audioElement) {
    let volume = 0;
    audioElement.volume = volume;
    const fadeInInterval = setInterval(() => {
        if (volume < 1) {
            volume += 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeInInterval);
        }
    }, 100);
}

function fadeOut(audioElement, callback) {
    let volume = audioElement.volume;
    const fadeOutInterval = setInterval(() => {
        if (volume > 0) {
            volume -= 0.1;
            audioElement.volume = volume;
        } else {
            clearInterval(fadeOutInterval);
            callback();
        }
    }, 100);
}

function playSFX(sfx) {
    if (sfx) {
        const sfxElement = new Audio(sfx);
        sfxElement.play().catch(err => console.error("SFX play error:", err));
    }
}

bgmElement?.addEventListener('timeupdate', () => {
    bgmPosition = bgmElement.currentTime;
});

function typeDialogue(text, textTickSFX, callback) {
    const dialogueText = document.getElementById('dialogue-text');
    if (!dialogueText) {
        console.error("Dialogue text element not found");
        return;
    }
    dialogueText.textContent = '';
    let i = 0;
    const typingSpeed = 60;
    const interval = setInterval(() => {
        if (i < text.length) {
            dialogueText.textContent += text.charAt(i);
            if (textTickSFX && text.charAt(i).trim() !== '') {
                const tickSFX = new Audio(textTickSFX);
                tickSFX.play().catch(err => console.error("Text tick SFX play error:", err));
            }
            i++;
        } else {
            clearInterval(interval);
            setTimeout(callback, 500);
        }
    }, typingSpeed);
}

function showChoices(choices) {
    const choicesContainer = document.getElementById('choices');
    if (!choicesContainer) {
        console.error("Choices container not found");
        return;
    }
    choicesContainer.style.display = 'block';
    choicesContainer.innerHTML = '';
    choices.forEach(choice => {
        const button = document.createElement('button');
        button.className = 'choice-button';
        button.textContent = choice.text;
        button.addEventListener('click', () => {
            choicesContainer.style.display = 'none';
            showScene(choice.nextScene);
        });
        choicesContainer.appendChild(button);
    });
}